package pangea.hiagent.document;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.document.Document;
import org.springframework.ai.reader.TextReader;
import org.springframework.ai.transformer.splitter.TokenTextSplitter;
import org.springframework.ai.vectorstore.VectorStore;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import pangea.hiagent.model.Agent;

import java.util.List;
import java.util.UUID;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.Resource;

/**
 * 文档管理服务
 * 负责创建和管理各Agent的知识库
 */
@Slf4j
@Service
public class DocumentManagementService {
    
    @Autowired(required = false)
    private VectorStore vectorStore;
    
    /**
     * 为客服助手创建知识库
     */
    public void createCustomerServiceKnowledgeBase() {
        if (vectorStore == null) {
            log.warn("VectorStore未配置，无法创建知识库");
            return;
        }
        
        try {
            String collectionId = "customer-service-kb";
            String knowledgeBaseContent = """
                客服助手知识库
                
                1. 服务理念
                我们致力于为客户提供友好、专业和高效的服务。始终以客户满意为目标，耐心解答客户问题。
                
                2. 常见问题处理
                - 产品咨询：详细介绍产品功能、规格和使用方法
                - 售后服务：指导客户如何申请退换货、维修等服务
                - 投诉处理：认真听取客户意见，及时反馈给相关部门
                
                3. 服务流程
                - 问候客户并询问需求
                - 理解客户问题并提供准确解答
                - 如需进一步处理，告知客户后续步骤和时间节点
                - 结束对话前确认客户是否还有其他问题
                """;
            
            // 创建文档
            Resource resource = new ByteArrayResource(knowledgeBaseContent.getBytes());
            TextReader textReader = new TextReader(resource);
            List<Document> documents = textReader.get();
            
            // 分割文档
            TokenTextSplitter textSplitter = new TokenTextSplitter();
            List<Document> splitDocuments = textSplitter.apply(documents);
            
            // 添加元数据
            for (int i = 0; i < splitDocuments.size(); i++) {
                Document originalDoc = splitDocuments.get(i);
                Document newDoc = Document.builder()
                    .id(UUID.randomUUID().toString())
                    .text(originalDoc.getText())
                    .metadata("collectionId", collectionId)
                    .metadata("agent", "客服助手")
                    .build();
                splitDocuments.set(i, newDoc);
            }
            
            // 存储到向量数据库
            vectorStore.add(splitDocuments);
            
            log.info("客服助手知识库创建完成，共{}个文档片段", splitDocuments.size());
        } catch (Exception e) {
            log.error("创建客服助手知识库失败", e);
        }
    }
    
    /**
     * 为技术支持创建知识库
     */
    public void createTechnicalSupportKnowledgeBase() {
        if (vectorStore == null) {
            log.warn("VectorStore未配置，无法创建知识库");
            return;
        }
        
        try {
            String collectionId = "technical-support-kb";
            String knowledgeBaseContent = """
                技术支持知识库
                
                1. 技术支持范围
                - 软件安装和配置指导
                - 硬件故障诊断和排除
                - 网络连接问题解决
                - 系统性能优化建议
                - 技术文档检索和代码解释
                
                2. 常见技术问题
                - 启动失败：检查电源连接、硬件状态和系统日志
                - 运行缓慢：分析资源占用情况，优化系统配置
                - 功能异常：确认软件版本，检查配置文件
                - 兼容性问题：核实系统环境，提供适配方案
                - 代码理解困难：使用代码解释工具分析代码逻辑
                
                3. 解决问题的方法论
                - 信息收集：详细了解问题现象、环境信息和操作步骤
                - 问题分析：根据现象判断可能的原因
                - 解决方案：提供具体的操作步骤和注意事项
                - 验证结果：确认问题是否得到解决
                - 文档检索：使用技术文档检索工具查找相关文档
                - 代码分析：使用代码解释工具理解复杂代码逻辑
                
                4. 工具使用指南
                - technicalDocumentationRetrieval: 用于检索技术文档，输入查询关键词即可
                - technicalCodeExplanation: 用于解释代码功能，输入代码内容和编程语言
                """;
            
            // 创建文档
            Resource resource = new ByteArrayResource(knowledgeBaseContent.getBytes());
            TextReader textReader = new TextReader(resource);
            List<Document> documents = textReader.get();
            
            // 分割文档
            TokenTextSplitter textSplitter = new TokenTextSplitter();
            List<Document> splitDocuments = textSplitter.apply(documents);
            
            // 添加元数据
            for (int i = 0; i < splitDocuments.size(); i++) {
                Document originalDoc = splitDocuments.get(i);
                Document newDoc = Document.builder()
                    .id(UUID.randomUUID().toString())
                    .text(originalDoc.getText())
                    .metadata("collectionId", collectionId)
                    .metadata("agent", "技术支持")
                    .build();
                splitDocuments.set(i, newDoc);
            }
            
            // 存储到向量数据库
            vectorStore.add(splitDocuments);
            
            log.info("技术支持知识库创建完成，共{}个文档片段", splitDocuments.size());
        } catch (Exception e) {
            log.error("创建技术支持知识库失败", e);
        }
    }
    
    /**
     * 为数据分析员创建知识库
     */
    public void createDataAnalysisKnowledgeBase() {
        if (vectorStore == null) {
            log.warn("VectorStore未配置，无法创建知识库");
            return;
        }
        
        try {
            String collectionId = "data-analysis-kb";
            String knowledgeBaseContent = """
                数据分析知识库
                
                1. 数据分析方法
                - 描述性分析：总结数据的基本特征
                - 探索性分析：发现数据中的模式和关系
                - 推断性分析：基于样本数据推断总体特征
                - 预测性分析：利用历史数据预测未来趋势
                
                2. 常用分析工具
                - Excel：适用于小型数据集的基本分析
                - Python：强大的数据分析和可视化工具
                - R：专门用于统计分析的编程语言
                - Tableau：直观的数据可视化平台
                
                3. 数据处理流程
                - 数据收集：确定数据来源和收集方法
                - 数据清洗：处理缺失值、异常值和重复数据
                - 数据转换：将数据转换为适合分析的格式
                - 数据分析：应用适当的分析方法得出结论
                - 结果呈现：以图表和报告形式展示分析结果
                
                4. 工具使用指南
                - calculator: 基础数学计算工具
                - chartGeneration: 图表生成工具，可用于生成柱状图、折线图、饼图等
                - statisticalCalculation: 统计计算工具，可用于计算基本统计信息、相关系数和线性回归分析
                """;
            
            // 创建文档
            Resource resource = new ByteArrayResource(knowledgeBaseContent.getBytes());
            TextReader textReader = new TextReader(resource);
            List<Document> documents = textReader.get();
            
            // 分割文档
            TokenTextSplitter textSplitter = new TokenTextSplitter();
            List<Document> splitDocuments = textSplitter.apply(documents);
            
            // 添加元数据
            for (int i = 0; i < splitDocuments.size(); i++) {
                Document originalDoc = splitDocuments.get(i);
                Document newDoc = Document.builder()
                    .id(UUID.randomUUID().toString())
                    .text(originalDoc.getText())
                    .metadata("collectionId", collectionId)
                    .metadata("agent", "数据分析员")
                    .build();
                splitDocuments.set(i, newDoc);
            }
            
            // 存储到向量数据库
            vectorStore.add(splitDocuments);
            
            log.info("数据分析员知识库创建完成，共{}个文档片段", splitDocuments.size());
        } catch (Exception e) {
            log.error("创建数据分析员知识库失败", e);
        }
    }
    
    /**
     * 为内容创作助手创建知识库
     */
    public void createContentCreationKnowledgeBase() {
        if (vectorStore == null) {
            log.warn("VectorStore未配置，无法创建知识库");
            return;
        }
        
        try {
            String collectionId = "content-creation-kb";
            String knowledgeBaseContent = """
                内容创作知识库
                
                1. 创作类型
                - 营销文案：突出产品卖点，激发购买欲望
                - 技术文档：准确描述功能，便于用户理解
                - 新闻报道：客观陈述事实，传递有价值信息
                - 社交媒体内容：简洁有趣，易于传播
                
                2. 创作原则
                - 目标明确：清楚了解内容的目标受众和传播目的
                - 结构清晰：合理安排内容结构，便于读者理解
                - 语言生动：使用恰当的修辞手法，增强表达效果
                - 信息准确：确保内容的真实性和可靠性
                
                3. 创作流程
                - 需求分析：了解创作背景、目标和要求
                - 资料收集：搜集相关素材和参考资料
                - 大纲制定：规划内容框架和重点章节
                - 初稿撰写：按照大纲完成初稿写作
                - 修改完善：检查内容质量，进行必要的修改
                
                4. 工具使用指南
                - search: 搜索相关资料和参考内容
                - writingStyleReference: 创作风格参考工具，提供各种写作风格的参考和指导
                - documentTemplate: 文档模板工具，提供各种类型的文档模板
                """;
            
            // 创建文档
            Resource resource = new ByteArrayResource(knowledgeBaseContent.getBytes());
            TextReader textReader = new TextReader(resource);
            List<Document> documents = textReader.get();
            
            // 分割文档
            TokenTextSplitter textSplitter = new TokenTextSplitter();
            List<Document> splitDocuments = textSplitter.apply(documents);
            
            // 添加元数据
            for (int i = 0; i < splitDocuments.size(); i++) {
                Document originalDoc = splitDocuments.get(i);
                Document newDoc = Document.builder()
                    .id(UUID.randomUUID().toString())
                    .text(originalDoc.getText())
                    .metadata("collectionId", collectionId)
                    .metadata("agent", "内容创作助手")
                    .build();
                splitDocuments.set(i, newDoc);
            }
            
            // 存储到向量数据库
            vectorStore.add(splitDocuments);
            
            log.info("内容创作助手知识库创建完成，共{}个文档片段", splitDocuments.size());
        } catch (Exception e) {
            log.error("创建内容创作助手知识库失败", e);
        }
    }
    
    /**
     * 为学习导师创建知识库
     */
    public void createLearningMentorKnowledgeBase() {
        if (vectorStore == null) {
            log.warn("VectorStore未配置，无法创建知识库");
            return;
        }
        
        try {
            String collectionId = "learning-mentor-kb";
            String knowledgeBaseContent = """
                学习导师知识库
                
                1. 学习方法指导
                - 主动学习：通过提问、讨论等方式积极参与学习过程
                - 分散学习：将学习内容分散到多个时间段，避免疲劳
                - 多样化学习：结合阅读、听讲、实践等多种方式
                - 反思总结：定期回顾所学内容，加深理解和记忆
                
                2. 学科辅导
                - 数学：注重逻辑思维训练，掌握解题方法和技巧
                - 语文：培养阅读理解能力，提高写作表达水平
                - 英语：加强听说读写四项技能的综合训练
                - 科学：理解科学原理，培养实验探究能力
                
                3. 学习规划
                - 目标设定：制定明确、可衡量的学习目标
                - 时间管理：合理安排学习时间，提高学习效率
                - 进度跟踪：定期检查学习进展，及时调整计划
                - 效果评估：通过测试和练习检验学习成果
                
                4. 工具使用指南
                - search: 搜索相关学习资料
                - studyPlanGeneration: 学习计划制定工具，可根据学习目标和时间安排制定个性化的学习计划
                - courseMaterialRetrieval: 课程资料检索工具，可检索和查询相关课程资料
                """;
            
            // 创建文档
            Resource resource = new ByteArrayResource(knowledgeBaseContent.getBytes());
            TextReader textReader = new TextReader(resource);
            List<Document> documents = textReader.get();
            
            // 分割文档
            TokenTextSplitter textSplitter = new TokenTextSplitter();
            List<Document> splitDocuments = textSplitter.apply(documents);
            
            // 添加元数据
            for (int i = 0; i < splitDocuments.size(); i++) {
                Document originalDoc = splitDocuments.get(i);
                Document newDoc = Document.builder()
                    .id(UUID.randomUUID().toString())
                    .text(originalDoc.getText())
                    .metadata("collectionId", collectionId)
                    .metadata("agent", "学习导师")
                    .build();
                splitDocuments.set(i, newDoc);
            }
            
            // 存储到向量数据库
            vectorStore.add(splitDocuments);
            
            log.info("学习导师知识库创建完成，共{}个文档片段", splitDocuments.size());
        } catch (Exception e) {
            log.error("创建学习导师知识库失败", e);
        }
    }
    
    /**
     * 为指定Agent创建知识库
     * @param agent Agent对象
     */
    public void createKnowledgeBaseForAgent(Agent agent) {
        if (agent == null) {
            log.warn("Agent对象为空，无法创建知识库");
            return;
        }
        
        switch (agent.getName()) {
            case "客服助手":
                createCustomerServiceKnowledgeBase();
                break;
            case "技术支持":
                createTechnicalSupportKnowledgeBase();
                break;
            case "数据分析员":
                createDataAnalysisKnowledgeBase();
                break;
            case "内容创作助手":
                createContentCreationKnowledgeBase();
                break;
            case "学习导师":
                createLearningMentorKnowledgeBase();
                break;
            default:
                log.warn("未找到{}对应的知識庫創建方法", agent.getName());
        }
    }
}