# 工具类参数配置机制实现方案

## 1. 设计目标

为项目中所有工具类实现参数配置功能，支持：
- 参数的UI动态配置展示
- 参数的数据库存储与读取
- 类似Spring Boot中@Value注解的机制实现参数管理
- 参数的默认值设置
- 数据库持久化存储与读取功能

## 2. 核心组件设计

### 2.1 自定义注解设计

创建`@ToolParam`注解，用于标记工具类中的配置参数：
```java
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
@Documented
public @interface ToolParam {
    String name() default "";
    String description() default "";
    String defaultValue() default "";
    String type() default "string";
    boolean required() default false;
    String group() default "default";
}
```

### 2.2 数据库实体设计

创建`ToolConfig`实体类，用于存储工具参数配置：
```java
@Entity
@Table(name = "tool_configs")
public class ToolConfig {
    @Id
    @GeneratedValue(strategy = GenerationType.UUID)
    private String id;
    
    private String toolName;
    private String paramName;
    private String paramValue;
    private String description;
    private String defaultValue;
    private String type;
    private boolean required;
    private String groupName;
    
    // getter和setter方法
}
```

### 2.3 参数配置服务

创建`ToolConfigService`服务类，用于处理参数的读取和保存：
```java
public interface ToolConfigService {
    Map<String, String> getToolParams(String toolName);
    String getParamValue(String toolName, String paramName);
    void saveParamValue(String toolName, String paramName, String paramValue);
    List<ToolConfig> getAllToolConfigs();
    ToolConfig getToolConfig(String toolName, String paramName);
    void saveToolConfig(ToolConfig toolConfig);
    void deleteToolConfig(String id);
}
```

### 2.4 参数配置处理器

创建`ToolParamProcessor`类，用于处理工具类中的`@ToolParam`注解：
```java
@Component
public class ToolParamProcessor {
    private final ToolConfigService toolConfigService;
    
    // 构造函数注入
    
    @PostConstruct
    public void processToolParams() {
        // 扫描所有带有@Component注解的工具类
        // 处理@ToolParam注解
        // 从数据库读取参数值并注入到工具类字段
    }
    
    public void injectParams(Object toolInstance) {
        // 为工具实例注入参数值
    }
}
```

### 2.5 参数配置控制器

创建`ToolConfigController`类，用于提供参数配置的REST API：
```java
@RestController
@RequestMapping("/api/tool-configs")
public class ToolConfigController {
    private final ToolConfigService toolConfigService;
    
    // 构造函数注入
    
    @GetMapping
    public List<ToolConfig> getAllToolConfigs() {
        // 返回所有工具配置
    }
    
    @GetMapping("/{toolName}")
    public Map<String, String> getToolParams(@PathVariable String toolName) {
        // 返回指定工具的配置参数
    }
    
    @PostMapping
    public ToolConfig saveToolConfig(@RequestBody ToolConfig toolConfig) {
        // 保存工具配置
    }
    
    @DeleteMapping("/{id}")
    public void deleteToolConfig(@PathVariable String id) {
        // 删除工具配置
    }
}
```

## 3. 实现步骤

### 3.1 创建核心组件

1. 创建`@ToolParam`注解
2. 创建`ToolConfig`实体类
3. 创建`ToolConfigRepository`接口
4. 创建`ToolConfigService`接口及实现类
5. 创建`ToolParamProcessor`类
6. 创建`ToolConfigController`类
7. 创建数据库初始化脚本，添加`tool_configs`表

### 3.2 修改工具类

以`FileProcessingTools.java`为例，修改其参数定义：

```java
@Slf4j
@Component
public class FileProcessingTools {
    
    // 支持的文本文件扩展名
    private static final List<String> TEXT_FILE_EXTENSIONS = Arrays.asList(
        ".txt", ".md", ".java", ".html", ".htm", ".css", ".js", ".json", 
        ".xml", ".yaml", ".yml", ".properties", ".sql", ".py", ".cpp", 
        ".c", ".h", ".cs", ".php", ".rb", ".go", ".rs", ".swift", ".kt", 
        ".scala", ".sh", ".bat", ".cmd", ".ps1", ".log", ".csv", ".ts", 
        ".jsx", ".tsx", ".vue", ".scss", ".sass", ".less"
    );
    
    // 支持的图片文件扩展名
    private static final List<String> IMAGE_FILE_EXTENSIONS = Arrays.asList(
        ".jpg", ".jpeg", ".png", ".gif", ".bmp", ".svg", ".webp", ".ico"
    );
    
    // 默认文件存储目录
    @ToolParam(
        name = "defaultStorageDir",
        description = "默认文件存储目录",
        defaultValue = "storage",
        type = "string",
        required = true,
        group = "file"
    )
    private String defaultStorageDir;
    
    // getter和setter方法
    
    // 其他方法保持不变
}
```

### 3.3 实现参数注入机制

在`ToolParamProcessor`类中实现参数注入逻辑，使用Spring的`BeanPostProcessor`接口，在Bean初始化后注入参数值：

```java
@Component
public class ToolParamProcessor implements BeanPostProcessor {
    private final ToolConfigService toolConfigService;
    
    // 构造函数注入
    
    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        if (bean.getClass().getPackage().getName().contains("pangea.hiagent.tools")) {
            injectParams(bean);
        }
        return bean;
    }
    
    private void injectParams(Object bean) {
        Class<?> clazz = bean.getClass();
        String toolName = clazz.getSimpleName();
        
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields) {
            if (field.isAnnotationPresent(ToolParam.class)) {
                ToolParam annotation = field.getAnnotation(ToolParam.class);
                String paramName = annotation.name().isEmpty() ? field.getName() : annotation.name();
                
                // 从数据库获取参数值，如果不存在则使用默认值
                String paramValue = toolConfigService.getParamValue(toolName, paramName);
                if (paramValue == null) {
                    paramValue = annotation.defaultValue();
                }
                
                // 设置字段值
                field.setAccessible(true);
                try {
                    // 根据字段类型转换参数值
                    if (field.getType() == String.class) {
                        field.set(bean, paramValue);
                    } else if (field.getType() == int.class || field.getType() == Integer.class) {
                        field.set(bean, Integer.parseInt(paramValue));
                    } else if (field.getType() == long.class || field.getType() == Long.class) {
                        field.set(bean, Long.parseLong(paramValue));
                    } else if (field.getType() == boolean.class || field.getType() == Boolean.class) {
                        field.set(bean, Boolean.parseBoolean(paramValue));
                    } else if (field.getType() == double.class || field.getType() == Double.class) {
                        field.set(bean, Double.parseDouble(paramValue));
                    }
                    // 可以扩展支持更多类型
                } catch (Exception e) {
                    // 处理异常
                }
            }
        }
    }
}
```

### 3.4 实现数据库初始化

创建`schema.sql`文件，添加`tool_configs`表的创建语句：

```sql
CREATE TABLE IF NOT EXISTS tool_configs (
    id VARCHAR(36) PRIMARY KEY,
    tool_name VARCHAR(255) NOT NULL,
    param_name VARCHAR(255) NOT NULL,
    param_value TEXT,
    description TEXT,
    default_value TEXT,
    type VARCHAR(50) NOT NULL,
    required BOOLEAN NOT NULL DEFAULT FALSE,
    group_name VARCHAR(100) NOT NULL DEFAULT 'default',
    created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY uk_tool_param (tool_name, param_name)
);
```

## 4. 扩展支持

### 4.1 UI配置界面支持

前端可以通过调用`/api/tool-configs`接口获取所有工具的配置参数，然后动态生成配置界面。

### 4.2 支持更多参数类型

可以扩展`ToolParamProcessor`类，支持更多参数类型，如枚举、数组、列表等。

### 4.3 支持参数验证

可以在`@ToolParam`注解中添加验证规则，如正则表达式、最小值、最大值等，然后在参数注入时进行验证。

## 5. 测试计划

1. 测试`@ToolParam`注解的基本功能
2. 测试参数的数据库存储与读取
3. 测试参数的默认值设置
4. 测试参数的类型转换
5. 测试参数的UI动态配置展示
6. 测试多个工具类的参数配置

## 6. 实现顺序

1. 创建核心组件（注解、实体、服务等）
2. 实现数据库初始化脚本
3. 实现参数注入机制
4. 修改现有工具类，添加`@ToolParam`注解
5. 实现参数配置控制器
6. 测试功能
7. 优化和扩展功能

## 7. 预期效果

通过本方案的实现，项目中的所有工具类都将支持参数的动态配置，用户可以通过UI界面方便地配置和管理工具参数，参数配置将持久化到数据库中，并且支持类似Spring Boot中@Value注解的机制实现参数管理。