# HiAgent定时器模块设计方案

## 1. 模块概述

### 1.1 功能简介
定时器模块是HiAgent系统的扩展组件，用于提供定时自动执行Agent任务的能力。该模块允许用户创建、编辑和管理多个独立定时器，每个定时器可配置不同的执行周期、目标Agent和提示词模板，实现自动化任务执行。

### 1.2 核心功能
- 多定时器管理：支持创建、编辑、删除多个独立定时器
- 灵活的周期配置：支持秒级精度的自定义执行周期
- Agent自动调用：定时触发指定Agent执行任务
- 提示词模板管理：支持预设提示词模板及动态参数替换
- 状态监控：实时显示定时器运行状态和执行历史

### 1.3 系统架构图

```
┌─────────────────────────────────────────────────────────────────────────┐
│                            HiAgent系统                                  │
├───────────────┬─────────────────────────────────────────────────────────┤
│   前端模块    │                       后端模块                          │
├───────────────┼─────────────────────────────────────────────────────────┤
│               │  ┌─────────────────┐   ┌─────────────────┐   ┌─────────┐ │
│  定时器管理   │  │  定时器API层    │   │  定时器服务层   │   │  任务    │ │
│  UI组件       │  │ (TimerController)│───│ (TimerService)  │───│ 调度器   │ │
│               │  └─────────────────┘   └─────────────────┘   │ (Quartz)│ │
│               │                                               └─────────┘ │
│               │                                                    │     │
│               │ ┌─────────────────┐   ┌─────────────────┐          │     │
│  提示词模板   │ │  提示词服务层   │   │  Agent服务层    │          │     │
│  管理组件     │ │ (PromptService)  │───│ (AgentService)  │◄─────────┘     │
│               │ └─────────────────┘   └─────────────────┘                │
│               │                                                           │
│               │ ┌─────────────────┐   ┌─────────────────┐                │
│  执行历史     │ │  历史记录API    │   │  历史记录服务   │                │
│  查看组件     │ │ (HistoryController)│───│ (HistoryService) │              │
│               │ └─────────────────┘   └─────────────────┘                │
└───────────────┴─────────────────────────────────────────────────────────┘
```

## 2. 数据结构设计

### 2.1 核心实体类

#### 2.1.1 定时器配置(TimerConfig)
```java
public class TimerConfig {
    private Long id;                      // 唯一标识
    private String name;                  // 定时器名称
    private String description;           // 描述
    private String cronExpression;        // Cron表达式（支持秒级）
    private boolean enabled;              // 启用状态
    private Long agentId;                 // 关联的Agent ID
    private String agentName;             // 关联的Agent名称
    private String promptTemplate;        // 提示词模板
    private String paramsJson;            // 动态参数配置（JSON格式）
    private Date createTime;              // 创建时间
    private Date updateTime;              // 更新时间
    private String createdBy;             // 创建人
}
```

#### 2.1.2 定时器执行历史(TimerExecutionHistory)
```java
public class TimerExecutionHistory {
    private Long id;                      // 唯一标识
    private Long timerId;                 // 关联的定时器ID
    private String timerName;             // 定时器名称
    private Date executionTime;           // 执行时间
    private boolean success;              // 执行结果
    private String result;                // 执行结果详情
    private String errorMessage;          // 错误信息
    private Long executionDuration;       // 执行时长（毫秒）
}
```

#### 2.1.3 提示词模板(PromptTemplate)
```java
public class PromptTemplate {
    private Long id;                      // 唯一标识
    private String name;                  // 模板名称
    private String description;           // 描述
    private String templateContent;       // 模板内容
    private String paramSchema;           // 参数Schema定义（JSON格式）
    private Date createTime;              // 创建时间
    private Date updateTime;              // 更新时间
    private String createdBy;             // 创建人
}
```

### 2.2 数据持久化
- 使用MySQL数据库存储定时器配置、执行历史和提示词模板
- 采用JPA/Hibernate框架实现ORM映射
- 表名设计：
  - `hiagent_timer_config`：存储定时器配置
  - `hiagent_timer_execution_history`：存储定时器执行历史
  - `hiagent_prompt_template`：存储提示词模板

## 3. 后端模块设计

### 3.1 模块结构

```
pangea.hiagent.timer/
├── controller/          # API控制器
│   ├── TimerController.java           # 定时器管理API
│   ├── TimerHistoryController.java    # 执行历史API
│   └── PromptTemplateController.java  # 提示词模板API
├── service/             # 业务逻辑层
│   ├── TimerService.java              # 定时器核心服务
│   ├── TimerHistoryService.java       # 执行历史服务
│   └── PromptTemplateService.java     # 提示词模板服务
├── scheduler/           # 任务调度层
│   ├── TimerScheduler.java            # 定时器调度器
│   └── TimerJob.java                  # 定时任务执行器
├── model/               # 数据模型
│   ├── TimerConfig.java               # 定时器配置实体
│   ├── TimerExecutionHistory.java     # 执行历史实体
│   └── PromptTemplate.java            # 提示词模板实体
├── dto/                 # 数据传输对象
│   ├── TimerConfigDto.java            # 定时器配置DTO
│   ├── TimerExecutionHistoryDto.java  # 执行历史DTO
│   └── PromptTemplateDto.java         # 提示词模板DTO
└── repository/          # 数据访问层
    ├── TimerConfigRepository.java     # 定时器配置DAO
    ├── TimerExecutionHistoryRepository.java  # 执行历史DAO
    └── PromptTemplateRepository.java  # 提示词模板DAO
```

### 3.2 核心服务设计

#### 3.2.1 TimerService
- **核心职责**：定时器的创建、编辑、删除、启用/禁用等管理操作
- **主要方法**：
  - `createTimer(TimerConfigDto config)`：创建定时器
  - `updateTimer(Long id, TimerConfigDto config)`：更新定时器
  - `deleteTimer(Long id)`：删除定时器
  - `enableTimer(Long id)`：启用定时器
  - `disableTimer(Long id)`：禁用定时器
  - `getTimerById(Long id)`：获取定时器详情
  - `listTimers()`：获取定时器列表

#### 3.2.2 TimerScheduler
- **核心职责**：管理定时任务的调度和执行
- **技术选型**：Quartz Scheduler（支持复杂Cron表达式和分布式部署）
- **主要功能**：
  - 动态添加/更新/删除定时任务
  - 支持秒级精度的任务调度
  - 任务执行状态监控
  - 故障恢复机制

#### 3.2.3 TimerJob
- **核心职责**：执行定时任务逻辑
- **主要流程**：
  1. 获取定时器配置
  2. 解析提示词模板并替换动态参数
  3. 调用AgentService执行任务
  4. 记录执行历史
  5. 处理执行结果

#### 3.2.4 PromptTemplateService
- **核心职责**：提示词模板的管理和参数替换
- **主要方法**：
  - `createTemplate(PromptTemplateDto template)`：创建提示词模板
  - `updateTemplate(Long id, PromptTemplateDto template)`：更新模板
  - `deleteTemplate(Long id)`：删除模板
  - `getTemplateById(Long id)`：获取模板详情
  - `listTemplates()`：获取模板列表
  - `renderTemplate(String templateContent, Map<String, Object> params)`：渲染模板，替换动态参数

### 3.3 API接口设计

| API路径 | 方法 | 功能描述 | 请求体 | 响应体 |
|---------|------|----------|--------|--------|
| /api/timers | POST | 创建定时器 | TimerConfigDto | TimerConfigDto |
| /api/timers/{id} | PUT | 更新定时器 | TimerConfigDto | TimerConfigDto |
| /api/timers/{id} | DELETE | 删除定时器 | - | ApiResponse |
| /api/timers/{id}/enable | POST | 启用定时器 | - | ApiResponse |
| /api/timers/{id}/disable | POST | 禁用定时器 | - | ApiResponse |
| /api/timers | GET | 获取定时器列表 | - | List<TimerConfigDto> |
| /api/timers/{id} | GET | 获取定时器详情 | - | TimerConfigDto |
| /api/timers/history | GET | 获取执行历史 | - | List<TimerExecutionHistoryDto> |
| /api/timers/{id}/history | GET | 获取指定定时器执行历史 | - | List<TimerExecutionHistoryDto> |
| /api/prompt-templates | POST | 创建提示词模板 | PromptTemplateDto | PromptTemplateDto |
| /api/prompt-templates/{id} | PUT | 更新提示词模板 | PromptTemplateDto | PromptTemplateDto |
| /api/prompt-templates/{id} | DELETE | 删除提示词模板 | - | ApiResponse |
| /api/prompt-templates | GET | 获取提示词模板列表 | - | List<PromptTemplateDto> |
| /api/prompt-templates/{id} | GET | 获取提示词模板详情 | - | PromptTemplateDto |

## 4. 前端模块设计

### 4.1 页面结构

#### 4.1.1 定时器管理页面(TimerManagement.vue)
- **布局**：
  - 顶部：操作按钮（创建定时器、刷新列表）
  - 中部：定时器列表表格
  - 底部：分页控件
- **表格列**：
  - 名称、描述、周期、关联Agent、状态、创建时间、操作（编辑、删除、启用/禁用）
- **交互**：
  - 点击"创建定时器"打开编辑弹窗
  - 点击"编辑"打开编辑弹窗
  - 点击"删除"弹出确认对话框
  - 点击"启用/禁用"切换定时器状态

#### 4.1.2 定时器编辑弹窗(TimerEditDialog.vue)
- **布局**：
  - 基本信息：名称、描述、创建人
  - 周期配置：Cron表达式输入框、可视化Cron编辑器
  - Agent配置：Agent下拉选择器
  - 提示词配置：提示词模板选择器、模板编辑区、参数配置区
- **交互**：
  - 支持直接输入Cron表达式或使用可视化编辑器
  - 实时验证Cron表达式有效性
  - 支持从模板库选择提示词模板
  - 支持在线编辑提示词模板
  - 支持动态添加/删除参数

#### 4.1.3 执行历史页面(TimerHistory.vue)
- **布局**：
  - 顶部：筛选条件（定时器ID、时间范围、执行结果）
  - 中部：执行历史表格
  - 底部：分页控件
- **表格列**：
  - 定时器名称、执行时间、执行结果、执行时长、操作（查看详情）
- **交互**：
  - 支持多条件筛选
  - 点击"查看详情"弹出执行详情对话框

### 4.2 组件关系图

```
┌─────────────────────────────────────────────────────────┐
│                   定时器管理页面                         │
│  (TimerManagement.vue)                                 │
├─────────────────────────────────────────────────────────┤
│  ┌─────────────────┐  ┌─────────────────┐  ┌───────────┐ │
│  │  定时器列表     │  │  定时器编辑弹窗  │  │  执行历史  │ │
│  │  (TimerList.vue)│  │(TimerEditDialog.vue)│(History.vue)│ │
│  └─────────────────┘  └─────────────────┘  └───────────┘ │
│           │                    │                          │
│           ▼                    ▼                          │
│  ┌─────────────────┐  ┌─────────────────┐                │
│  │  Cron编辑器     │  │  提示词模板管理  │                │
│  │ (CronEditor.vue)│  │(PromptTemplate.vue)│              │
│  └─────────────────┘  └─────────────────┘                │
└─────────────────────────────────────────────────────────┘
```

### 4.3 核心组件设计

#### 4.3.1 Cron编辑器组件(CronEditor.vue)
- **功能**：提供可视化的Cron表达式编辑界面
- **支持的Cron类型**：
  - 每秒执行
  - 每分钟执行
  - 每小时执行
  - 每日执行
  - 每周执行
  - 每月执行
  - 自定义Cron表达式
- **交互**：
  - 提供下拉选择和输入框组合
  - 实时生成和验证Cron表达式
  - 支持Cron表达式的解析和可视化展示

#### 4.3.2 提示词模板组件(PromptTemplate.vue)
- **功能**：管理提示词模板和参数配置
- **特性**：
  - 支持从模板库选择模板
  - 支持在线编辑模板内容
  - 支持动态添加/删除参数
  - 支持参数类型校验
  - 实时预览渲染后的提示词

### 4.4 状态管理
- 使用Pinia进行状态管理
- 定义`timerStore`存储定时器相关状态：
  - 定时器列表
  - 当前编辑的定时器
  - 执行历史
  - 提示词模板列表
  - 加载状态和错误信息

## 5. Agent调用机制

### 5.1 调用流程

```
┌─────────────────┐     ┌─────────────────┐     ┌─────────────────┐
│  TimerJob触发   │────▶│  TimerService   │────▶│  AgentService   │
└─────────────────┘     └─────────────────┘     └─────────────────┘
                                                           │
                                                           ▼
                                               ┌─────────────────┐
                                               │  ReActService   │
                                               └─────────────────┘
                                                           │
                                                           ▼
                                               ┌─────────────────┐
                                               │   LLM模型调用   │
                                               └─────────────────┘
                                                           │
                                                           ▼
                                               ┌─────────────────┐
                                               │  执行结果记录   │
                                               └─────────────────┘
```

### 5.2 提示词渲染与参数替换

#### 5.2.1 模板语法
采用Mustache风格的模板语法，支持：
- 简单变量替换：`{{variable}}`
- 条件判断：`{{#condition}}内容{{/condition}}`
- 循环：`{{#list}}{{.}}{{/list}}`
- 默认值：`{{variable || '默认值'}}`

#### 5.2.2 内置参数
系统自动提供以下内置参数：
- `{{currentTime}}`：当前时间（ISO格式）
- `{{currentDate}}`：当前日期
- `{{timerName}}`：定时器名称
- `{{executionCount}}`：执行次数

#### 5.2.3 自定义参数
用户可在创建定时器时配置自定义参数，支持以下类型：
- 字符串
- 数字
- 布尔值
- 日期时间
- JSON对象

## 6. 与现有系统集成

### 6.1 集成点

| 集成模块 | 集成方式 | 主要功能 |
|---------|----------|----------|
| Agent服务 | 直接调用 | 获取Agent列表、执行Agent任务 |
| 认证系统 | JWT令牌 | 确保API访问安全 |
| 日志系统 | 日志记录 | 记录定时器执行日志 |
| 数据库 | JPA/Hibernate | 数据持久化 |

### 6.2 依赖关系

| 依赖模块 | 版本要求 | 用途 |
|---------|----------|------|
| Spring Boot | 2.7.x+ | 基础框架 |
| Quartz Scheduler | 2.3.x+ | 任务调度 |
| Vue 3 | 3.4.x+ | 前端框架 |
| Element Plus | 2.4.x+ | UI组件库 |
| MySQL | 8.0.x+ | 数据库 |

## 7. 性能与扩展性设计

### 7.1 性能优化
- **任务调度优化**：使用Quartz的线程池机制，限制并发执行的任务数量
- **数据库优化**：
  - 为执行历史表添加索引
  - 定期清理过期的执行历史记录
  - 使用分页查询减少数据传输量
- **缓存优化**：
  - 缓存Agent列表和提示词模板
  - 缓存定时器配置，减少数据库查询

### 7.2 扩展性设计
- **模块化设计**：采用模块化架构，便于后续扩展新功能
- **插件化支持**：支持自定义任务执行器插件
- **分布式支持**：基于Quartz的分布式特性，支持多实例部署
- **API扩展性**：提供RESTful API，便于与其他系统集成

## 8. 安全设计

### 8.1 认证与授权
- 使用JWT令牌保护API访问
- 基于角色的访问控制（RBAC）
- 限制定时器创建和管理权限

### 8.2 数据安全
- 敏感数据加密存储
- 执行历史数据脱敏处理
- 防止SQL注入和XSS攻击

### 8.3 任务安全
- 限制单个定时器的执行频率
- 设置任务执行超时机制
- 防止任务无限循环执行

## 9. 监控与日志

### 9.1 监控指标
- 定时器执行成功率
- 平均执行时间
- 任务队列长度
- 系统资源使用率

### 9.2 日志记录
- 记录定时器的创建、编辑、删除操作
- 记录定时器的启用、禁用状态变化
- 记录每次任务执行的详细日志
- 记录执行过程中的错误信息

## 10. 部署与维护

### 10.1 部署方式
- 与HiAgent主系统一起部署
- 支持Docker容器化部署
- 支持Kubernetes集群部署

### 10.2 配置管理
- 定时器相关配置集中管理
- 支持动态调整配置参数
- 配置变更日志记录

### 10.3 维护建议
- 定期清理过期的执行历史记录
- 监控系统资源使用率
- 定期备份定时器配置数据
- 建立完善的告警机制

## 11. 总结

HiAgent定时器模块是一个功能完整、设计合理的扩展组件，它为HiAgent系统提供了定时自动执行Agent任务的能力。该模块采用了模块化、可扩展的设计架构，支持多定时器管理、灵活的周期配置、Agent自动调用和提示词模板管理等核心功能。

该设计方案充分考虑了系统的性能、安全性和扩展性，能够满足不同场景下的定时任务需求。通过与现有系统的无缝集成，定时器模块将进一步提升HiAgent系统的自动化能力和用户体验。