package pangea.hiagent.websocket;

import lombok.extern.slf4j.Slf4j;
import org.springframework.web.socket.*;
import pangea.hiagent.workpanel.playwright.PlaywrightManager;
import pangea.hiagent.common.utils.UserUtils;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * WebSocket连接管理器
 * 负责处理WebSocket连接的建立、关闭和错误处理
 */
@Slf4j
public class WebSocketConnectionManager {
    // 存储连接的前端客户端（线程安全）
    private final ConcurrentMap<WebSocketSession, String> clients = new ConcurrentHashMap<>();
    
    // 连接数限制
    private static final int MAX_CONNECTIONS_PER_USER = 5; // 每用户最大连接数
    
    // 用户连接计数
    private final ConcurrentMap<String, Integer> userConnections = new ConcurrentHashMap<>();
    
    private final PlaywrightManager playwrightManager;
    
    public WebSocketConnectionManager(PlaywrightManager playwrightManager) {
        this.playwrightManager = playwrightManager;
    }
    
    /**
     * 处理客户端连接建立
     */
    public void handleConnectionEstablished(WebSocketSession session, DomSyncService domSyncService) {
        // 从会话属性中获取用户ID（来自JWT认证）
        String userId = (String) session.getAttributes().get("userId");
        if (userId == null || userId.isEmpty()) {
            log.warn("WebSocket连接缺少有效的用户认证信息，使用默认用户ID");
            userId = "unknown-user";
            // 不再拒绝连接，而是使用默认用户ID
        }
        
        // 检查连接数限制
        Integer currentConnections = userConnections.getOrDefault(userId, 0);
        if (currentConnections >= MAX_CONNECTIONS_PER_USER) {
            try {
                session.close(CloseStatus.POLICY_VIOLATION.withReason("超过最大连接数限制"));
                return;
            } catch (Exception e) {
                log.error("关闭WebSocket会话失败", e);
            }
        }
        
        // 增加用户连接数
        userConnections.put(userId, currentConnections + 1);
        
        clients.put(session, session.getId());
        
        // 为用户初始化专用的浏览器上下文
        boolean initSuccess = domSyncService.initUserBrowserContext(userId);
        if (!initSuccess) {
            log.error("为用户 {} 初始化浏览器上下文失败", userId);
            try {
                session.close(CloseStatus.SERVER_ERROR.withReason("初始化浏览器上下文失败"));
            } catch (Exception e) {
                log.error("关闭WebSocket会话失败", e);
            }
            return;
        }
        
        // 创建消息发送器
        BinaryMessageSender messageSender = new BinaryMessageSender(this);
        
        // 初始化页面监听器
        boolean listenerInitSuccess = domSyncService.initPageListener(messageSender);
        if (!listenerInitSuccess) {
            log.warn("页面监听器初始化失败");
        }
    }
    
    /**
     * 处理客户端连接关闭
     */
    public void handleConnectionClosed(WebSocketSession session, DomSyncService domSyncService) {
        clients.remove(session);
        
        // 从会话属性中获取用户ID（来自JWT认证）
        String userId = (String) session.getAttributes().get("userId");
        if (userId == null || userId.isEmpty()) {
            // 如果没有有效的用户ID，尝试从SecurityContext获取
            userId = UserUtils.getCurrentUserIdStatic();
            if (userId == null || userId.isEmpty()) {
                // 如果仍然无法获取用户ID，使用默认值
                userId = "unknown-user";
                log.warn("WebSocket连接关闭时缺少有效的用户认证信息，使用默认用户ID: {}", userId);
            }
        }
        
        // 减少用户连接数
        Integer currentConnections = userConnections.getOrDefault(userId, 0);
        if (currentConnections > 0) {
            userConnections.put(userId, currentConnections - 1);
        }
        
        // 如果该用户没有其他连接，释放其浏览器上下文
        if (currentConnections <= 1) {
            domSyncService.releaseUserResources(userId, playwrightManager);
        }
    }
    
    /**
     * 处理传输错误
     */
    public void handleTransportError(WebSocketSession session, Throwable exception) {
        clients.remove(session);
        log.warn("客户端传输错误：{}，错误信息：{}", session.getId(), exception.getMessage());
    }
    
    /**
     * 获取当前连接的客户端数量
     */
    public int getClientCount() {
        return clients.size();
    }
    
    /**
     * 获取指定会话
     */
    public WebSocketSession getSession(String sessionId) {
        return clients.entrySet().stream()
                .filter(entry -> entry.getValue().equals(sessionId))
                .map(entry -> entry.getKey())
                .findFirst()
                .orElse(null);
    }
    
    /**
     * 广播消息给所有客户端
     */
    public void broadcastMessage(BinaryMessage message, BinaryMessageSender messageSender) {
        int successCount = 0;
        int failureCount = 0;
        
        for (WebSocketSession client : clients.keySet()) {
            try {
                if (client.isOpen()) {
                    client.sendMessage(message);
                    successCount++;
                } else {
                    log.warn("客户端 [{}] 连接已关闭", client.getId());
                    clients.remove(client);
                    failureCount++;
                }
            } catch (Exception e) {
                log.error("发送消息给客户端 [{}] 失败: {}", client.getId(), e.getMessage(), e);
                clients.remove(client);
                failureCount++;
            }
        }
        
        log.debug("消息广播完成: 成功发送给 {} 个客户端, 失败 {} 个客户端", successCount, failureCount);
    }
    
    /**
     * 销毁资源
     */
    public void destroy() {
        try {
            // 关闭所有客户端连接
            for (WebSocketSession session : clients.keySet()) {
                try {
                    if (session.isOpen()) {
                        session.close(CloseStatus.NORMAL);
                    }
                } catch (Exception e) {
                    log.warn("关闭WebSocket会话失败 [{}]", session.getId(), e);
                }
            }
            clients.clear();
        } catch (Exception e) {
            log.error("清理客户端连接失败", e);
        }
    }
}