package pangea.hiagent.common.config;

import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.access.expression.method.DefaultMethodSecurityExpressionHandler;
import org.springframework.security.access.expression.method.MethodSecurityExpressionHandler;
import org.springframework.security.config.annotation.method.configuration.EnableMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;
import org.springframework.web.cors.CorsConfiguration;
import org.springframework.web.cors.CorsConfigurationSource;
import org.springframework.web.cors.UrlBasedCorsConfigurationSource;

import pangea.hiagent.web.service.AgentService;
import pangea.hiagent.web.service.TimerService;
import pangea.hiagent.security.DefaultPermissionEvaluator;
import pangea.hiagent.security.JwtAuthenticationFilter;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;

@Slf4j
@Configuration
@EnableWebSecurity
@EnableMethodSecurity(prePostEnabled = true)
public class SecurityConfig {
    
    private final JwtAuthenticationFilter jwtAuthenticationFilter;
    private final AgentService agentService;
    private final TimerService timerService;
    
    public SecurityConfig(JwtAuthenticationFilter jwtAuthenticationFilter, AgentService agentService, TimerService timerService) {
        this.jwtAuthenticationFilter = jwtAuthenticationFilter;
        this.agentService = agentService;
        this.timerService = timerService;
    }
    
    /**
     * 密码编码器
     */
    @Bean
    public PasswordEncoder passwordEncoder() {
        return new BCryptPasswordEncoder();
    }
    
    /**
     * 方法安全表达式处理器
     */
    @Bean
    public MethodSecurityExpressionHandler methodSecurityExpressionHandler() {
        DefaultMethodSecurityExpressionHandler expressionHandler = new DefaultMethodSecurityExpressionHandler();
        // 创建带有AgentService和TimerService的权限评估器
        DefaultPermissionEvaluator permissionEvaluator = new DefaultPermissionEvaluator(agentService, timerService);
        expressionHandler.setPermissionEvaluator(permissionEvaluator);
        return expressionHandler;
    }
    
    /**
     * CORS配置
     */
    @Bean
    public CorsConfigurationSource corsConfigurationSource() {
        CorsConfiguration configuration = new CorsConfiguration();
        configuration.setAllowedOrigins(Collections.singletonList("*"));
        configuration.setAllowedMethods(Arrays.asList("GET", "POST", "PUT", "DELETE", "OPTIONS", "PATCH"));
        configuration.setAllowedHeaders(Collections.singletonList("*"));
        configuration.setExposedHeaders(Arrays.asList("Authorization", "Content-Type"));
        configuration.setMaxAge(3600L);
        configuration.setAllowCredentials(false);
        
        UrlBasedCorsConfigurationSource source = new UrlBasedCorsConfigurationSource();
        source.registerCorsConfiguration("/**", configuration);
        return source;
    }
    
    /**
     * Security过滤链配置
     */
    @Bean
    public SecurityFilterChain filterChain(HttpSecurity http) throws Exception {
        http
                // 禁用CSRF
                .csrf(csrf -> csrf.disable())
                // 启用CORS
                .cors(cors -> cors.configurationSource(corsConfigurationSource()))
                // 设置session创建策略为无状态
                .sessionManagement(session -> session.sessionCreationPolicy(SessionCreationPolicy.STATELESS))
                // 配置请求授权
                .authorizeHttpRequests(authz -> authz
                        // WebSocket端点 - 由握手拦截器处理认证，不需要通过Spring Security过滤链
                        .requestMatchers("/ws/**").permitAll()
                        // OAuth2 相关端点公开访问
                        .requestMatchers("/api/v1/auth/oauth2/**").permitAll()
                        // OAuth2提供商管理端点需要认证（仅管理员可访问）
                        .requestMatchers("/api/v1/auth/oauth2/providers/**").authenticated()
                        // 公开端点
                        .requestMatchers(
                                "/api/v1/auth/**",
                                "/swagger-ui.html",
                                "/swagger-ui/**",
                                "/v3/api-docs/**",
                                "/h2-console/**",
                                "/redoc.html",
                                "/error",
                                "/api/v1/proxy/**"  // 将proxy接口设为公开访问
                        ).permitAll()
                        // Agent相关端点 - 需要认证
                        .requestMatchers("/api/v1/agent/**").authenticated()
                        // 特别允许timeline-events端点通过认证检查（由JWT过滤器处理）
                        .requestMatchers("/api/v1/agent/timeline-events").authenticated()
                        // 工具相关端点 - 需要认证
                        .requestMatchers("/api/v1/tools/**").authenticated()
                        // 所有其他请求需要认证
                        .anyRequest().authenticated()
                )
                // 异常处理
                .exceptionHandling(exception -> exception
                        .authenticationEntryPoint((request, response, authException) -> {
                            // 更全面地检查响应是否已经提交
                            if (response.isCommitted()) {
                                log.warn("响应已经提交，无法处理认证异常: {}", request.getRequestURI());
                                return;
                            }
                            
                            try {
                                // 对于SSE端点的特殊处理
                                boolean isStreamEndpoint = request.getRequestURI().contains("/api/v1/agent/chat-stream");
                                boolean isTimelineEndpoint = request.getRequestURI().contains("/api/v1/agent/timeline-events");
                                
                                if (isStreamEndpoint || isTimelineEndpoint) {
                                    // 对于SSE端点，发送SSE格式的错误事件
                                    response.setContentType("text/event-stream;charset=UTF-8");
                                    response.setCharacterEncoding("UTF-8");
                                    response.getWriter().write("event: error\ndata: {\"error\": \"未授权访问\", \"timestamp\": " + System.currentTimeMillis() + "}\n\n");
                                    response.getWriter().flush();
                                    // 确保响应被正确提交
                                    if (!response.isCommitted()) {
                                        response.flushBuffer();
                                    }
                                    return;
                                }
                                
                                response.setStatus(401);
                                response.setContentType("application/json;charset=UTF-8");
                                response.getWriter().write("{\"code\":401,\"message\":\"未授权访问\",\"timestamp\":" + System.currentTimeMillis() + "}");
                                response.getWriter().flush();
                                // 确保响应被正确提交
                                if (!response.isCommitted()) {
                                    response.flushBuffer();
                                }
                            } catch (IOException e) {
                                log.error("发送认证错误响应失败: {}", request.getRequestURI(), e);
                                // 如果在发送错误响应时发生IO异常，确保不会导致未处理的异常
                            } catch (Exception e) {
                                log.error("处理认证异常时发生未知错误: {}", request.getRequestURI(), e);
                            }
                        })
                        .accessDeniedHandler((request, response, accessDeniedException) -> {
                            // 更全面地检查响应是否已经提交
                            if (response.isCommitted()) {
                                log.warn("响应已经提交，无法处理访问拒绝异常: {}", request.getRequestURI());
                                return;
                            }
                            
                            try {
                                // 对于SSE端点的特殊处理
                                boolean isStreamEndpoint = request.getRequestURI().contains("/api/v1/agent/chat-stream");
                                boolean isTimelineEndpoint = request.getRequestURI().contains("/api/v1/agent/timeline-events");
                                
                                if (isStreamEndpoint || isTimelineEndpoint) {
                                    // 对于SSE端点，发送SSE格式的错误事件
                                    response.setContentType("text/event-stream;charset=UTF-8");
                                    response.setCharacterEncoding("UTF-8");
                                    response.getWriter().write("event: error\ndata: {\"error\": \"访问被拒绝\", \"timestamp\": " + System.currentTimeMillis() + "}\n\n");
                                    response.getWriter().flush();
                                    // 确保响应被正确提交
                                    if (!response.isCommitted()) {
                                        response.flushBuffer();
                                    }
                                    return;
                                }
                                
                                response.setStatus(403);
                                response.setContentType("application/json;charset=UTF-8");
                                response.getWriter().write("{\"code\":403,\"message\":\"访问被拒绝\",\"timestamp\":" + System.currentTimeMillis() + "}");
                                response.getWriter().flush();
                                // 确保响应被正确提交
                                if (!response.isCommitted()) {
                                    response.flushBuffer();
                                }
                            } catch (IOException e) {
                                log.error("发送访问拒绝响应失败: {}", request.getRequestURI(), e);
                                // 如果在发送错误响应时发生IO异常，确保不会导致未处理的异常
                            } catch (Exception e) {
                                log.error("处理访问拒绝异常时发生未知错误: {}", request.getRequestURI(), e);
                            }
                        })
                )
                // 添加JWT认证过滤器
                .addFilterBefore(jwtAuthenticationFilter, UsernamePasswordAuthenticationFilter.class)
                // 配置X-Frame-Options头部，允许同源iframe嵌入
                .headers(headers -> headers
                        .frameOptions(frameOptions -> frameOptions
                                .sameOrigin()
                        )
                );
        
        return http.build();
    }
}