package pangea.hiagent.core;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.Tool;
import pangea.hiagent.core.AgentToolManager;

import java.util.List;

/**
 * 提示词服务类
 * 负责构建和管理系统提示词
 */
@Slf4j
@Service
public class PromptService {
    
    @Autowired
    private AgentToolManager agentToolManager;
    
    /**
     * 构建系统提示词 - 根据Agent配置的工具动态生成
     * @param agent Agent对象
     * @return 系统提示词
     */
    public String buildSystemPrompt(Agent agent) {
        // 如果Agent配置了自定义系统提示词，优先使用
        if (agent.getSystemPrompt() != null && !agent.getSystemPrompt().isEmpty()) {
            return agent.getSystemPrompt();
        }
        
        try {
            // 获取Agent配置的可用工具列表
            List<Tool> agentTools = agentToolManager.getAvailableTools(agent);
            
            // 如果没有工具，直接返回默认提示词
            if (agentTools.isEmpty()) {
                return getDefaultSystemPrompt();
            }
            
            // 构建工具描述部分
            String toolsDescription = agentToolManager.buildToolsDescription(agentTools);
            String toolsList = buildToolsList(agentTools);
            
            // 构建默认系统提示词，包含动态生成的工具信息
            return buildDefaultSystemPrompt(toolsDescription, toolsList);
        } catch (Exception e) {
            log.error("构建系统提示词时发生错误", e);
            // 返回默认的系统提示词
            return getDefaultSystemPrompt();
        }
    }
    
    /**
     * 构建工具名称列表（用于Action的可选值）
     * @param tools 工具列表
     * @return 工具名称列表，逗号分隔
     */
    private String buildToolsList(List<Tool> tools) {
        if (tools.isEmpty()) {
            return "(no tools available)";
        }
        
        return tools.stream()
                .map(Tool::getName)
                .collect(java.util.stream.Collectors.joining(", "));
    }
    
    /**
     * 构建默认系统提示词
     * @param toolsDescription 工具描述
     * @param toolsList 工具列表
     * @return 系统提示词
     */
    private String buildDefaultSystemPrompt(String toolsDescription, String toolsList) {
        return "You are a helpful AI assistant that can use tools to help answer questions.\n" +
                "You have access to the following tools:\n" +
                toolsDescription +
                "\n\nTo use a tool, please use the following format:\n" +
                "Thought: Do I need to use a tool? Yes\n" +
                "Action: the action to take, should be one of [" + toolsList + "]\n" +
                "Action Input: the input to the action\n" +
                "Observation: the result of the action\n\n" +
                "When you have a response to say to the Human, or if you do not need to use a tool, you MUST use the format:\n" +
                "Thought: Do I need to use a tool? No\n" +
                "Final Answer: [your response here]";
    }
    
    /**
     * 获取默认系统提示词（当工具加载失败时使用）
     * @return 默认系统提示词
     */
    public String getDefaultSystemPrompt() {
        return "You are a helpful AI assistant that can use tools to help answer questions.\n" +
                "You have access to the following tools:\n" +
                "1. getCurrentDateTime - Get current date and time\n" +
                "2. add, subtract, multiply, divide - Basic math operations\n" +
                "3. processString - String processing\n" +
                "4. readFile, writeFile - File operations\n" +
                "5. getWeather - Get weather information\n\n" +
                "To use a tool, please use the following format:\n" +
                "Thought: Do I need to use a tool? Yes\n" +
                "Action: the action to take\n" +
                "Action Input: the input to the action\n" +
                "Observation: the result of the action\n\n" +
                "When you have a response to say to the Human, or if you do not need to use a tool, you MUST use the format:\n" +
                "Thought: Do I need to use a tool? No\n" +
                "Final Answer: [your response here]";
    }
}