package pangea.hiagent.prompt;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.Tool;
import pangea.hiagent.service.ToolService;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 提示词服务类
 * 负责构建和管理系统提示词
 */
@Slf4j
@Service
public class PromptService {
    
    @Autowired
    private ToolService toolService;
    
    /**
     * 构建系统提示词 - 根据Agent配置的工具动态生成
     * @param agent Agent对象
     * @return 系统提示词
     */
    public String buildSystemPrompt(Agent agent) {
        // 如果Agent配置了自定义系统提示词，优先使用
        if (agent.getSystemPrompt() != null && !agent.getSystemPrompt().isEmpty()) {
            return agent.getSystemPrompt();
        }
        
        try {
            // 获取Agent配置的可用工具列表
            List<Tool> agentTools = getAvailableTools(agent);
            
            // 如果没有工具，直接返回默认提示词
            if (agentTools.isEmpty()) {
                return getDefaultSystemPrompt();
            }
            
            // 构建工具描述部分
            String toolsDescription = buildToolsDescription(agentTools);
            String toolsList = buildToolsList(agentTools);
            
            // 构建默认系统提示词，包含动态生成的工具信息
            return buildDefaultSystemPrompt(toolsDescription, toolsList);
        } catch (Exception e) {
            log.error("构建系统提示词时发生错误", e);
            // 返回默认的系统提示词
            return getDefaultSystemPrompt();
        }
    }
    
    /**
     * 获取Agent可用的工具列表
     * @param agent Agent对象
     * @return 工具列表
     */
    private List<Tool> getAvailableTools(Agent agent) {
        try {
            // 获取Agent所有者的所有活跃工具
            List<Tool> allTools = toolService.getUserToolsByStatus(agent.getOwner(), "active");
            if (allTools == null || allTools.isEmpty()) {
                log.warn("Agent: {} 没有配置可用的工具", agent.getId());
                return List.of();
            }
            
            // 如果Agent配置了特定的工具列表，则只返回配置的工具
            List<String> toolNames = agent.getToolNames();
            if (toolNames != null && !toolNames.isEmpty()) {
                // 根据工具名称筛选工具
                return filterToolsByName(allTools, toolNames);
            }
            
            return allTools;
        } catch (Exception e) {
            log.error("获取Agent可用工具时发生错误", e);
            return List.of();
        }
    }
    
    /**
     * 根据工具名称筛选工具
     * @param allTools 所有工具
     * @param toolNames 工具名称列表
     * @return 筛选后的工具列表
     */
    private List<Tool> filterToolsByName(List<Tool> allTools, List<String> toolNames) {
        return allTools.stream()
            .filter(tool -> toolNames.contains(tool.getName()))
            .collect(Collectors.toList());
    }
    
    /**
     * 构建工具描述文本
     * @param tools 工具列表
     * @return 工具描述文本
     */
    private String buildToolsDescription(List<Tool> tools) {
        if (tools.isEmpty()) {
            return "（暂无可用工具）";
        }
        
        StringBuilder description = new StringBuilder();
        for (int i = 0; i < tools.size(); i++) {
            Tool tool = tools.get(i);
            description.append(i + 1).append(". ");
            description.append(tool.getName());
            if (hasValue(tool.getDisplayName())) {
                description.append(" - ").append(tool.getDisplayName());
            }
            if (hasValue(tool.getDescription())) {
                description.append(" - ").append(tool.getDescription());
            }
            description.append("\n");
        }
        
        return description.toString();
    }
    
    /**
     * 检查字符串是否有值
     * @param value 字符串值
     * @return 是否有值
     */
    private boolean hasValue(String value) {
        return value != null && !value.isEmpty();
    }
    
    /**
     * 构建工具名称列表（用于Action的可选值）
     * @param tools 工具列表
     * @return 工具名称列表，逗号分隔
     */
    private String buildToolsList(List<Tool> tools) {
        if (tools.isEmpty()) {
            return "(no tools available)";
        }
        
        return tools.stream()
                .map(Tool::getName)
                .collect(Collectors.joining(", "));
    }
    
    /**
     * 构建默认系统提示词
     * @param toolsDescription 工具描述
     * @param toolsList 工具列表
     * @return 系统提示词
     */
    private String buildDefaultSystemPrompt(String toolsDescription, String toolsList) {
        return "You are a helpful AI assistant that can use tools to help answer questions.\n" +
                "You have access to the following tools:\n" +
                toolsDescription +
                "\n\nTo use a tool, please use the following format:\n" +
                "Thought: Do I need to use a tool? Yes\n" +
                "Action: the action to take, should be one of [" + toolsList + "]\n" +
                "Action Input: the input to the action\n" +
                "Observation: the result of the action\n\n" +
                "When you have a response to say to the Human, or if you do not need to use a tool, you MUST use the format:\n" +
                "Thought: Do I need to use a tool? No\n" +
                "Final Answer: [your response here]";
    }
    
    /**
     * 获取默认系统提示词（当工具加载失败时使用）
     * @return 默认系统提示词
     */
    public String getDefaultSystemPrompt() {
        return "You are a helpful AI assistant that can use tools to help answer questions.\n" +
                "You have access to the following tools:\n" +
                "1. getCurrentDateTime - Get current date and time\n" +
                "2. add, subtract, multiply, divide - Basic math operations\n" +
                "3. processString - String processing\n" +
                "4. readFile, writeFile - File operations\n" +
                "5. getWeather - Get weather information\n\n" +
                "To use a tool, please use the following format:\n" +
                "Thought: Do I need to use a tool? Yes\n" +
                "Action: the action to take\n" +
                "Action Input: the input to the action\n" +
                "Observation: the result of the action\n\n" +
                "When you have a response to say to the Human, or if you do not need to use a tool, you MUST use the format:\n" +
                "Thought: Do I need to use a tool? No\n" +
                "Final Answer: [your response here]";
    }
}