package pangea.hiagent.memory;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.memory.ChatMemory;
import org.springframework.ai.chat.messages.Message;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * 混合ChatMemory实现
 * 支持Caffeine和Redis两种缓存机制，并提供配置选项来启用或禁用这两种缓存功能
 */
@Slf4j
@Component
public class HybridChatMemory implements ChatMemory {
    
    @Autowired(required = false)
    private CaffeineChatMemory caffeineChatMemory;
    
    @Autowired(required = false)
    private RedisChatMemory redisChatMemory;
    
    // 是否启用Caffeine缓存
    @Value("${app.chat-memory.caffeine.enabled:true}")
    private boolean caffeineEnabled;
    
    // 是否启用Redis缓存
    @Value("${app.chat-memory.redis.enabled:true}")
    private boolean redisEnabled;
    
    @Override
    public void add(String conversationId, List<Message> messages) {
        boolean success = false;
        Exception lastException = null;
        
        // 尝试保存到Caffeine缓存
        if (caffeineEnabled && caffeineChatMemory != null) {
            try {
                caffeineChatMemory.add(conversationId, messages);
                success = true;
                log.debug("成功将消息保存到Caffeine缓存");
            } catch (Exception e) {
                lastException = e;
                log.warn("保存消息到Caffeine缓存时发生错误: {}", e.getMessage());
            }
        }
        
        // 尝试保存到Redis缓存
        if (redisEnabled && redisChatMemory != null) {
            try {
                redisChatMemory.add(conversationId, messages);
                success = true;
                log.debug("成功将消息保存到Redis缓存");
            } catch (Exception e) {
                lastException = e;
                log.warn("保存消息到Redis缓存时发生错误: {}", e.getMessage());
            }
        }
        
        // 如果两种缓存都失败了，抛出异常
        if (!success && lastException != null) {
            log.error("保存消息到所有缓存时都失败了", lastException);
            throw new RuntimeException("Failed to save messages to any cache", lastException);
        }
    }
    
    @Override
    public List<Message> get(String conversationId, int lastN) {
        // 优先从Caffeine缓存获取
        if (caffeineEnabled && caffeineChatMemory != null) {
            try {
                List<Message> messages = caffeineChatMemory.get(conversationId, lastN);
                if (messages != null && !messages.isEmpty()) {
                    log.debug("从Caffeine缓存获取到消息");
                    return messages;
                }
            } catch (Exception e) {
                log.warn("从Caffeine缓存获取消息时发生错误: {}", e.getMessage());
            }
        }
        
        // 从Redis缓存获取
        if (redisEnabled && redisChatMemory != null) {
            try {
                List<Message> messages = redisChatMemory.get(conversationId, lastN);
                if (messages != null && !messages.isEmpty()) {
                    log.debug("从Redis缓存获取到消息");
                    // 同步到Caffeine缓存以提高下次访问速度
                    if (caffeineEnabled && caffeineChatMemory != null) {
                        try {
                            caffeineChatMemory.add(conversationId, messages);
                            log.debug("已将Redis缓存的数据同步到Caffeine缓存");
                        } catch (Exception e) {
                            log.warn("同步数据到Caffeine缓存时发生错误: {}", e.getMessage());
                        }
                    }
                    return messages;
                }
            } catch (Exception e) {
                log.warn("从Redis缓存获取消息时发生错误: {}", e.getMessage());
            }
        }
        
        // 都没有获取到数据，返回空列表
        return new ArrayList<>();
    }
    
    @Override
    public void clear(String conversationId) {
        boolean success = false;
        Exception lastException = null;
        
        // 清除Caffeine缓存
        if (caffeineEnabled && caffeineChatMemory != null) {
            try {
                caffeineChatMemory.clear(conversationId);
                success = true;
                log.debug("成功清除Caffeine缓存中的会话数据");
            } catch (Exception e) {
                lastException = e;
                log.warn("清除Caffeine缓存中的会话数据时发生错误: {}", e.getMessage());
            }
        }
        
        // 清除Redis缓存
        if (redisEnabled && redisChatMemory != null) {
            try {
                redisChatMemory.clear(conversationId);
                success = true;
                log.debug("成功清除Redis缓存中的会话数据");
            } catch (Exception e) {
                lastException = e;
                log.warn("清除Redis缓存中的会话数据时发生错误: {}", e.getMessage());
            }
        }
        
        // 如果两种缓存都清除失败，抛出异常
        if (!success && lastException != null) {
            log.error("清除所有缓存中的会话数据时都失败了", lastException);
            throw new RuntimeException("Failed to clear conversation from any cache", lastException);
        }
    }
}