package pangea.hiagent.agent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.client.ChatClient;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.ai.chat.model.StreamingChatModel;
import org.springframework.stereotype.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import pangea.hiagent.model.Agent;
import pangea.hiagent.service.AgentService;
import pangea.hiagent.dto.WorkPanelEvent;
import pangea.hiagent.rag.RagService;
import pangea.hiagent.tool.DefaultReactExecutor;
import pangea.hiagent.tool.ReactCallback;
import pangea.hiagent.tool.ReactExecutor;
import pangea.hiagent.memory.MemoryService;
import pangea.hiagent.workpanel.IWorkPanelDataCollector;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * 基于Spring AI ChatClient的ReAct Service类
 * 负责实现ReAct Agent的核心逻辑，使用Spring AI的标准工具调用机制
 * 集成工作面板数据收集器，捕获思考、行动、观察、结束等步骤
 */
@Slf4j
@Service
public class ReActService {
    
    // 添加默认构造函数以支持测试
    public ReActService() {
    }    
    @Autowired
    private AgentService agentService;
    
    @Autowired
    private RagService ragService;
    
    // 注入所有带@Component注解的工具类
    @Autowired
    @Lazy
    private List<Object> allTools;
    
    @Autowired
    private IWorkPanelDataCollector workPanelCollector;
    
    @Autowired
    private MemoryService memoryService;
    
    @Autowired
    private ReactCallback defaultReactCallback;
    
    @Autowired
    private ReactExecutor defaultReactExecutor;
    
    /**
     * 根据工具名称筛选工具实例
     * @param toolNames 工具名称集合
     * @return 筛选后的工具实例列表
     */
    private List<Object> filterToolsByNames(Set<String> toolNames) {
        log.debug("开始筛选工具，工具名称集合: {}", toolNames);
        
        if (toolNames == null || toolNames.isEmpty()) {
            log.debug("工具名称集合为空，返回所有工具");
            return allTools;
        }
        
        List<Object> filteredTools = allTools.stream()
            .filter(tool -> {
                // 获取工具类名（不含包名）
                String className = tool.getClass().getSimpleName();
                log.debug("检查工具类: {}", className);
                
                // 检查类名是否匹配
                boolean isMatch = toolNames.contains(className) ||
                    toolNames.stream().anyMatch(name ->
                        className.toLowerCase().contains(name.toLowerCase()));
                
                if (isMatch) {
                    log.debug("工具 {} 匹配成功", className);
                }
                
                return isMatch;
            })
            .collect(Collectors.toList());
            
        log.debug("筛选完成，返回 {} 个工具", filteredTools.size());
        return filteredTools;
    }
    
    /**
     * 处理用户请求的主方法（同步方式）
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @return 处理结果
     */
    public String processRequest(Agent agent, String userMessage) {
        return processRequestWithUserId(agent, userMessage, null);
    }
    
    /**
     * 处理用户请求的主方法（同步方式）- 支持显式传递userId
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @param userId 用户ID（可选）
     * @return 处理结果
     */
    public String processRequestWithUserId(Agent agent, String userMessage, String userId) {
        log.info("开始处理ReAct Agent请求，Agent ID: {}, 用户消息: {}", agent.getId(), userMessage);
        
        try {
            // 为每个用户-Agent组合创建唯一的会话ID
            String sessionId = memoryService.generateSessionId(agent, userId);
            
            // 添加用户消息到ChatMemory
            memoryService.addUserMessageToMemory(sessionId, userMessage);
            
            // 检查是否启用RAG并尝试RAG增强
            String ragResponse = tryRagEnhancement(agent, userMessage);
            if (ragResponse != null) {
                // 触发最终答案回调
                if (defaultReactCallback != null) {
                    defaultReactCallback.onFinalAnswer(ragResponse);
                }
                return ragResponse;
            }
            
            // 准备执行环境
            ChatClient client = prepareChatClient(agent);
            List<Object> tools = prepareTools(agent);
            
            // 添加自定义回调到ReAct执行器
            addReactCallbackIfNeeded();
            
            // 使用ReAct执行器执行流程，传递Agent对象以支持记忆功能
            String finalAnswer = executeReactProcess(client, userMessage, tools, agent);
            
            // 将助理回复添加到ChatMemory
            memoryService.addAssistantMessageToMemory(sessionId, finalAnswer);
            
            return finalAnswer;
        } catch (Exception e) {
            // 检查是否是401 Unauthorized错误
            if (isUnauthorizedError(e)) {
                log.error("LLM返回401未授权错误: {}", e.getMessage());
                return " 请配置API密钥";
            } else {
                log.error("处理ReAct请求时发生错误", e);
                return "处理请求时发生错误: " + e.getMessage();
            }
        }
    }
    
    /**
     * 准备ChatClient
     * @param agent Agent对象
     * @return ChatClient实例
     */
    private ChatClient prepareChatClient(Agent agent) {
        // 根据Agent配置获取对应的ChatModel
        ChatModel chatModel = agentService.getChatModelForAgent(agent);
        log.info("获取ChatModel成功: {}", chatModel.getClass().getName());
        
        // 使用获取的ChatModel构建ChatClient
        return ChatClient.builder(chatModel).build();
    }
    
    /**
     * 准备工具列表
     * @param agent Agent对象
     * @return 工具列表
     */
    private List<Object> prepareTools(Agent agent) {
        log.info("准备工具列表，Agent ID: {}, Agent名称: {}", agent.getId(), agent.getName());
        
        // 获取Agent配置的工具名称集合
        Set<String> toolNames = agent.getToolNameSet();
        log.info("Agent配置的工具名称集合: {}", toolNames);
        
        // 根据工具名称筛选工具实例
        List<Object> tools = filterToolsByNames(toolNames);
        log.info("筛选后的工具数量: {}", tools.size());
        
        // 记录工具详情
        tools.forEach(tool -> log.debug("工具类: {}", tool.getClass().getSimpleName()));
        
        return tools;
    }
    
    /**
     * 添加ReAct回调（如果需要）
     */
    private void addReactCallbackIfNeeded() {
        if (defaultReactExecutor != null && defaultReactCallback != null) {
            defaultReactExecutor.addReactCallback(defaultReactCallback);
        }
    }
    
    /**
     * 执行ReAct流程
     * @param client ChatClient实例
     * @param userMessage 用户消息
     * @param tools 工具列表
     * @param agent Agent对象
     * @return 最终答案
     */
    private String executeReactProcess(ChatClient client, String userMessage, List<Object> tools, Agent agent) {
        if (defaultReactExecutor instanceof DefaultReactExecutor) {
            return ((DefaultReactExecutor) defaultReactExecutor).executeWithAgent(client, userMessage, tools, agent);
        } else {
            return defaultReactExecutor.execute(client, userMessage, tools);
        }
    }
    
    /**
     * 尝试RAG增强
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @return RAG增强结果，如果没有启用或没有结果则返回null
     */
    private String tryRagEnhancement(Agent agent, String userMessage) {
        if (Boolean.TRUE.equals(agent.getEnableRag())) {
            log.info("Agent启用RAG功能，尝试RAG增强");
            try {
                String ragResponse = ragService.ragQa(agent, userMessage);
                
                // 如果RAG有结果，直接返回
                if (ragResponse != null && !ragResponse.isEmpty()) {
                    log.info("RAG增强返回结果");
                    return saveRagResponseToMemory(agent, ragResponse);
                }
            } catch (Exception e) {
                log.error("RAG增强过程中发生错误", e);
                // RAG失败不应该影响正常的ReAct流程，继续执行
            }
        }
        return null;
    }
    
    /**
     * 保存响应到ChatMemory
     * @param agent Agent对象
     * @param response 响应内容
     * @return 响应内容
     */
    private String saveResponseToMemory(Agent agent, String response) {
        try {
            String sessionId = memoryService.generateSessionId(agent);
            // 将响应添加到ChatMemory
            memoryService.addAssistantMessageToMemory(sessionId, response);
            return response;
        } catch (Exception e) {
            log.error("保存响应到内存时发生错误", e);
            return response; // 即使保存失败也返回响应
        }
    }
    
    /**
     * 保存RAG响应到ChatMemory (已废弃，使用saveResponseToMemory替代)
     * @param agent Agent对象
     * @param ragResponse RAG响应
     * @return RAG响应
     */
    @Deprecated
    private String saveRagResponseToMemory(Agent agent, String ragResponse) {
        return saveResponseToMemory(agent, ragResponse);
    }
    
    /**
     * 处理流式错误
     */
    private void handleStreamError(AtomicBoolean isCompleted, Consumer<String> tokenConsumer, String errorMessage) {
        // 确保只处理一次错误
        if (!isCompleted.getAndSet(true) && tokenConsumer != null) {
            try {
                // 记录详细错误日志
                log.error("流式处理错误: {}", errorMessage);
                
                // 同时将错误信息记录到工作面板
                if (workPanelCollector != null) {
                    try {
                        workPanelCollector.recordLog("流式处理错误: " + errorMessage, "error");
                    } catch (Exception e) {
                        log.debug("记录错误到工作面板失败: {}", e.getMessage());
                    }
                }
                
                // 发送错误信息给客户端
                tokenConsumer.accept("[ERROR] " + errorMessage);
            } catch (Exception e) {
                log.error("发送错误消息时发生异常", e);
            }
        }
    }
    
    /**
     * 获取流式模型
     */
    private StreamingChatModel getStreamingChatModel(Agent agent) {
        try {
            ChatModel chatModel = agentService.getChatModelForAgent(agent);
            if (!(chatModel instanceof StreamingChatModel)) {
                log.warn("模型不支持流式输出: {}", chatModel.getClass().getName());
                return null;
            }
            return (StreamingChatModel) chatModel;
        } catch (Exception e) {
            log.error("获取流式模型失败: {}", e.getMessage(), e);
            return null;
        }
    }
    
    /**
     * 流式处理ReAct Agent请求
     * 
     * 优化后的实现采用更直接的流式处理方式，确保与普通Agent流式处理保持一致的行为
     * 核心优化点：
     * 1. 简化了Consumer包装逻辑，减少不必要的复杂性
     * 2. 统一了onComplete回调机制，确保前端能正确接收到完整内容
     * 3. 增强了错误处理机制，提供更清晰的错误信息
     * 4. 支持对话历史记忆功能
     *
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @param tokenConsumer token处理回调函数（前端的TokenConsumerWithCompletion实现）
     */
    public void processRequestStream(Agent agent, String userMessage, Consumer<String> tokenConsumer) {
        processRequestStreamWithUserId(agent, userMessage, tokenConsumer, null);
    }
    
    /**
     * 流式处理ReAct Agent请求 - 支持显式传递userId
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @param tokenConsumer token处理回调函数（前端的TokenConsumerWithCompletion实现）
     * @param userId 用户ID（可选）
     */
    public void processRequestStreamWithUserId(Agent agent, String userMessage, Consumer<String> tokenConsumer, String userId) {
        AtomicBoolean isCompleted = new AtomicBoolean(false);
        try {
            log.info("开始流式处理ReAct Agent请求，Agent ID: {}, 用户消息: {}", agent.getId(), userMessage);
            
            // 检查用户消息是否为空
            if (userMessage == null || userMessage.trim().isEmpty()) {
                String errorMsg = "用户消息不能为空";
                log.error(errorMsg);
                handleStreamError(isCompleted, tokenConsumer, errorMsg);
                return;
            }
            
            // 为每个用户-Agent组合创建唯一的会话ID
            String sessionId = memoryService.generateSessionId(agent, userId);
            
            // 添加用户消息到ChatMemory
            memoryService.addUserMessageToMemory(sessionId, userMessage);
            
            // 初始化工作面板
            initializeWorkPanel(agent);
            
            // 获取流式模型
            StreamingChatModel streamingChatModel = getStreamingChatModel(agent);
            if (streamingChatModel == null) {
                String errorMsg = "当前模型不支持流式输出，请检查Agent配置";
                log.error(errorMsg);
                handleStreamError(isCompleted, tokenConsumer, errorMsg);
                return;
            }
            
            // 准备执行环境
            ChatClient client = ChatClient.builder((ChatModel) streamingChatModel).build();
            List<Object> tools = prepareTools(agent);
            
            // 添加自定义回调到ReAct执行器
            addReactCallbackIfNeeded();
            
            // 直接传递tokenConsumer给ReAct执行器，简化处理逻辑
            // ReAct执行器会负责处理token和onComplete回调
            // 传递Agent对象以支持记忆功能
            executeReactStreamProcess(client, userMessage, tools, tokenConsumer, agent);
            
            log.debug("流式执行完成");
            
        } catch (Exception e) {
            String errorMsg = "流式处理ReAct请求时发生错误: " + e.getMessage();
            // 检查是否是401 Unauthorized错误
            if (isUnauthorizedError(e)) {
                log.error("LLM返回401未授权错误: {}", e.getMessage());
                errorMsg = " 请配置API密钥";
            } else {
                log.error(errorMsg, e);
            }
            handleStreamError(isCompleted, tokenConsumer, errorMsg);
        }
    }
    
    /**
     * 执行ReAct流式流程
     * @param client ChatClient实例
     * @param userMessage 用户消息
     * @param tools 工具列表
     * @param tokenConsumer token处理回调函数
     * @param agent Agent对象
     */
    private void executeReactStreamProcess(ChatClient client, String userMessage, List<Object> tools, Consumer<String> tokenConsumer, Agent agent) {
        if (defaultReactExecutor instanceof DefaultReactExecutor) {
            ((DefaultReactExecutor) defaultReactExecutor).executeStreamWithAgent(
                client, userMessage, tools, tokenConsumer, agent);
        } else {
            defaultReactExecutor.executeStream(client, userMessage, tools, tokenConsumer);
        }
    }
    
    /**
     * 初始化工作面板
     * @param agent Agent对象
     */
    private void initializeWorkPanel(Agent agent) {
        if (workPanelCollector != null) {
            try {
                workPanelCollector.clear();
                workPanelCollector.recordLog("开始处理Agent请求: " + agent.getName(), "info");
            } catch (Exception e) {
                log.error("初始化工作面板时发生错误", e);
            }
        }
    }
    
    /**
     * 为流式处理设置工作面板数据收集器
     */
    public void setWorkPanelEventSubscriber(Consumer<WorkPanelEvent> consumer) {
        // 订阅工作面板事件，用于实时推送
        if (workPanelCollector != null && consumer != null) {
            workPanelCollector.subscribe(consumer);
        }
    }    
    /**
     * 获取工作面板数据收集器
     */
    public IWorkPanelDataCollector getWorkPanelCollector() {
        return workPanelCollector;
    }
    
    /**
     * 判断异常是否为401未授权错误
     * @param e 异常对象
     * @return 是否为401错误
     */
    private boolean isUnauthorizedError(Throwable e) {
        if (e == null) {
            return false;
        }
        
        // 检查异常消息中是否包含401 Unauthorized
        String message = e.getMessage();
        if (message != null && (message.contains("401 Unauthorized") || message.contains("Unauthorized"))) {
            return true;
        }
        
        // 递归检查cause
        return isUnauthorizedError(e.getCause());
    }

}