package pangea.hiagent.web.controller;

import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import pangea.hiagent.web.service.PromptTemplateService;
import pangea.hiagent.web.service.TimerService;
import pangea.hiagent.web.dto.ApiResponse;
import pangea.hiagent.web.dto.TimerConfigDto;
import pangea.hiagent.web.dto.PromptTemplateDto;
import pangea.hiagent.common.exception.ErrorCode;
import pangea.hiagent.model.TimerConfig;
import pangea.hiagent.model.PromptTemplate;
import pangea.hiagent.common.utils.UserUtils;

import jakarta.validation.Valid;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 定时器API控制器
 * 提供定时器的增删改查和管理功能
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/timer")
public class TimerController {

    private final TimerService timerService;
    private final PromptTemplateService promptTemplateService;

    public TimerController(TimerService timerService, PromptTemplateService promptTemplateService) {
        this.timerService = timerService;
        this.promptTemplateService = promptTemplateService;
    }

    // ==================== 定时器管理API ====================

    /**
     * 创建定时器
     */
    @PostMapping
    public ApiResponse<TimerConfigDto> createTimer(@Valid @RequestBody TimerConfigDto timerConfigDto) {
        try {
            String userId = getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }

            log.info("用户 {} 开始创建定时器: {}", userId, timerConfigDto.getName());
            
            // 转换DTO为实体
            TimerConfig timerConfig = convertToEntity(timerConfigDto);
            timerConfig.setCreatedBy(userId);
            
            TimerConfig created = timerService.createTimer(timerConfig);
            log.info("用户 {} 成功创建定时器: {} (ID: {})", userId, created.getName(), created.getId());
            
            return ApiResponse.success(convertToDto(created), "创建定时器成功");
        } catch (IllegalArgumentException e) {
            log.error("创建定时器失败 - 参数验证错误: {}", e.getMessage());
            return ApiResponse.error(ErrorCode.PARAMETER_ERROR.getCode(), "创建定时器失败: " + e.getMessage());
        } catch (Exception e) {
            log.error("创建定时器失败", e);
            Throwable cause = e.getCause();
            String errorMsg = "创建定时器失败: " + e.getMessage();
            if (cause != null) {
                errorMsg += " (原因: " + cause.getMessage() + ")";
            }
            return ApiResponse.error(4001, errorMsg);
        }
    }

    /**
     * 更新定时器
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'TimerConfig', 'write')")
    @PutMapping("/{id}")
    public ApiResponse<TimerConfigDto> updateTimer(@PathVariable String id, @Valid @RequestBody TimerConfigDto timerConfigDto) {
        try {
            String userId = getCurrentUserId();
            log.info("用户 {} 开始更新定时器: {}", userId, id);
            
            TimerConfig timerConfig = convertToEntity(timerConfigDto);
            timerConfig.setId(id);
            
            TimerConfig updated = timerService.updateTimer(timerConfig);
            log.info("用户 {} 成功更新定时器: {}", userId, updated.getId());
            
            return ApiResponse.success(convertToDto(updated), "更新定时器成功");
        } catch (IllegalArgumentException e) {
            log.error("更新定时器失败 - 参数验证错误: {}", e.getMessage());
            return ApiResponse.error(ErrorCode.PARAMETER_ERROR.getCode(), "更新定时器失败: " + e.getMessage());
        } catch (Exception e) {
            log.error("更新定时器失败", e);
            Throwable cause = e.getCause();
            String errorMsg = "更新定时器失败: " + e.getMessage();
            if (cause != null) {
                errorMsg += " (原因: " + cause.getMessage() + ")";
            }
            return ApiResponse.error(4001, errorMsg);
        }
    }

    /**
     * 删除定时器
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'TimerConfig', 'delete')")
    @DeleteMapping("/{id}")
    public ApiResponse<Void> deleteTimer(@PathVariable String id) {
        try {
            String userId = getCurrentUserId();
            log.info("用户 {} 开始删除定时器: {}", userId, id);
            
            timerService.deleteTimer(id);
            log.info("用户 {} 成功删除定时器: {}", userId, id);
            
            return ApiResponse.success(null, "删除定时器成功");
        } catch (Exception e) {
            log.error("删除定时器失败", e);
            return ApiResponse.error(4001, "删除定时器失败: " + e.getMessage());
        }
    }

    /**
     * 获取定时器详情
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'TimerConfig', 'read')")
    @GetMapping("/{id}")
    public ApiResponse<TimerConfigDto> getTimer(@PathVariable String id) {
        try {
            TimerConfig timerConfig = timerService.getTimerById(id);
            if (timerConfig == null) {
                return ApiResponse.error(4001, "定时器不存在");
            }
            return ApiResponse.success(convertToDto(timerConfig));
        } catch (Exception e) {
            log.error("获取定时器详情失败", e);
            return ApiResponse.error(4001, "获取定时器详情失败: " + e.getMessage());
        }
    }

    /**
     * 获取定时器列表
     */
    @GetMapping
    public ApiResponse<List<TimerConfigDto>> listTimers() {
        try {
            String userId = getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }
            
            List<TimerConfig> timers = timerService.listTimersByCreatedBy(userId);
            List<TimerConfigDto> timerDtos = timers.stream()
                    .map(this::convertToDto)
                    .collect(Collectors.toList());
            
            return ApiResponse.success(timerDtos);
        } catch (Exception e) {
            log.error("获取定时器列表失败", e);
            return ApiResponse.error(4001, "获取定时器列表失败: " + e.getMessage());
        }
    }

    /**
     * 启用定时器
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'TimerConfig', 'write')")
    @PostMapping("/{id}/enable")
    public ApiResponse<Void> enableTimer(@PathVariable String id) {
        try {
            String userId = getCurrentUserId();
            log.info("用户 {} 开始启用定时器: {}", userId, id);
            
            timerService.enableTimer(id);
            log.info("用户 {} 成功启用定时器: {}", userId, id);
            
            return ApiResponse.success(null, "启用定时器成功");
        } catch (Exception e) {
            log.error("启用定时器失败", e);
            return ApiResponse.error(4001, "启用定时器失败: " + e.getMessage());
        }
    }

    /**
     * 禁用定时器
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'TimerConfig', 'write')")
    @PostMapping("/{id}/disable")
    public ApiResponse<Void> disableTimer(@PathVariable String id) {
        try {
            String userId = getCurrentUserId();
            log.info("用户 {} 开始禁用定时器: {}", userId, id);
            
            timerService.disableTimer(id);
            log.info("用户 {} 成功禁用定时器: {}", userId, id);
            
            return ApiResponse.success(null, "禁用定时器成功");
        } catch (Exception e) {
            log.error("禁用定时器失败", e);
            return ApiResponse.error(4001, "禁用定时器失败: " + e.getMessage());
        }
    }



    // ==================== 提示词模板API ====================

    /**
     * 创建提示词模板
     */
    @PostMapping("/prompt-template")
    public ApiResponse<PromptTemplateDto> createPromptTemplate(@RequestBody PromptTemplateDto templateDto) {
        try {
            String userId = getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }

            log.info("用户 {} 开始创建提示词模板: {}", userId, templateDto.getName());
            
            PromptTemplate template = convertToEntity(templateDto);
            template.setCreatedBy(userId);
            
            PromptTemplate created = promptTemplateService.createTemplate(template);
            log.info("用户 {} 成功创建提示词模板: {} (ID: {})", userId, created.getName(), created.getId());
            
            return ApiResponse.success(convertToDto(created), "创建提示词模板成功");
        } catch (Exception e) {
            log.error("创建提示词模板失败", e);
            return ApiResponse.error(4001, "创建提示词模板失败: " + e.getMessage());
        }
    }

    /**
     * 获取提示词模板列表
     */
    @GetMapping("/prompt-template")
    public ApiResponse<List<PromptTemplateDto>> listPromptTemplates() {
        try {
            List<PromptTemplate> templates = promptTemplateService.listTemplates();
            List<PromptTemplateDto> templateDtos = templates.stream()
                    .map(this::convertToDto)
                    .collect(Collectors.toList());
            
            return ApiResponse.success(templateDtos);
        } catch (Exception e) {
            log.error("获取提示词模板列表失败", e);
            return ApiResponse.error(4001, "获取提示词模板列表失败: " + e.getMessage());
        }
    }

    // ==================== 私有辅助方法 ====================

    /**
     * 获取当前认证用户ID
     */
    private String getCurrentUserId() {
        return UserUtils.getCurrentUserId();
    }

    /**
     * 转换TimerConfig实体为DTO
     */
    private TimerConfigDto convertToDto(TimerConfig timerConfig) {
        if (timerConfig == null) {
            return null;
        }
        
        return TimerConfigDto.builder()
                .id(timerConfig.getId())
                .name(timerConfig.getName())
                .description(timerConfig.getDescription())
                .cronExpression(timerConfig.getCronExpression())
                .enabled(timerConfig.getEnabled())
                .agentId(timerConfig.getAgentId())
                .agentName(timerConfig.getAgentName())
                .promptTemplate(timerConfig.getPromptTemplate())
                .paramsJson(timerConfig.getParamsJson())
                .lastExecutionTime(timerConfig.getLastExecutionTime())
                .nextExecutionTime(timerConfig.getNextExecutionTime())
                .createdAt(timerConfig.getCreatedAt())
                .updatedAt(timerConfig.getUpdatedAt())
                .createdBy(timerConfig.getCreatedBy())
                .build();
    }

    /**
     * 转换TimerConfigDto为实体
     */
    private TimerConfig convertToEntity(TimerConfigDto timerConfigDto) {
        if (timerConfigDto == null) {
            return null;
        }
        
        TimerConfig timerConfig = new TimerConfig();
        timerConfig.setId(timerConfigDto.getId());
        timerConfig.setName(timerConfigDto.getName());
        timerConfig.setDescription(timerConfigDto.getDescription());
        timerConfig.setCronExpression(timerConfigDto.getCronExpression());
        timerConfig.setEnabled(timerConfigDto.getEnabled());
        timerConfig.setAgentId(timerConfigDto.getAgentId());
        timerConfig.setAgentName(timerConfigDto.getAgentName());
        timerConfig.setPromptTemplate(timerConfigDto.getPromptTemplate());
        timerConfig.setParamsJson(timerConfigDto.getParamsJson());
        timerConfig.setLastExecutionTime(timerConfigDto.getLastExecutionTime());
        timerConfig.setNextExecutionTime(timerConfigDto.getNextExecutionTime());
        
        return timerConfig;
    }

    /**
     * 转换PromptTemplate实体为DTO
     */
    private PromptTemplateDto convertToDto(PromptTemplate template) {
        if (template == null) {
            return null;
        }
        
        return PromptTemplateDto.builder()
                .id(template.getId())
                .name(template.getName())
                .description(template.getDescription())
                .templateContent(template.getTemplateContent())
                .paramSchema(template.getParamSchema())
                .templateType(template.getTemplateType())
                .isSystem(template.getIsSystem())
                .createdAt(template.getCreatedAt())
                .updatedAt(template.getUpdatedAt())
                .createdBy(template.getCreatedBy())
                .build();
    }

    /**
     * 转换PromptTemplateDto为实体
     */
    private PromptTemplate convertToEntity(PromptTemplateDto templateDto) {
        if (templateDto == null) {
            return null;
        }
        
        PromptTemplate template = new PromptTemplate();
        template.setId(templateDto.getId());
        template.setName(templateDto.getName());
        template.setDescription(templateDto.getDescription());
        template.setTemplateContent(templateDto.getTemplateContent());
        template.setParamSchema(templateDto.getParamSchema());
        template.setTemplateType(templateDto.getTemplateType());
        template.setIsSystem(templateDto.getIsSystem());
        
        return template;
    }
}
