package pangea.hiagent.web.controller;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.model.Tool;
import pangea.hiagent.web.dto.ApiResponse;
import pangea.hiagent.web.service.ToolService;

import org.springframework.web.bind.annotation.*;


import java.util.List;

/**
 * 工具API控制器
 * 提供工具的增删改查功能
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/tools")
@Tag(name = "工具管理", description = "工具管理相关API")
public class ToolController {
    
    private final ToolService toolService;
    
    public ToolController(ToolService toolService) {
        this.toolService = toolService;
    }
    
    /**
     * 获取当前认证用户ID
     * @return 用户ID
     */
    private String getCurrentUserId() {
        return UserUtils.getCurrentUserId();
    }
    
    /**
     * 创建工具
     */
    @PostMapping
    @Operation(summary = "创建工具", description = "创建一个新的工具")
    public ApiResponse<Tool> createTool(@RequestBody Tool tool) {
        try {
            String userId = getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }
            
            Tool created = toolService.createTool(tool, userId);
            return ApiResponse.success(created, "创建工具成功");
        } catch (Exception e) {
            log.error("创建工具失败", e);
            return ApiResponse.error(4001, "创建工具失败: " + e.getMessage());
        }
    }
    
    /**
     * 更新工具
     */
    @PutMapping("/{id}")
    @Operation(summary = "更新工具", description = "更新指定ID的工具")
    public ApiResponse<Tool> updateTool(@PathVariable String id, @RequestBody Tool tool) {
        try {
            String userId = getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }
            
            tool.setId(id);
            Tool updated = toolService.updateTool(tool, userId);
            return ApiResponse.success(updated, "更新工具成功");
        } catch (Exception e) {
            log.error("更新工具失败", e);
            return ApiResponse.error(4001, "更新工具失败: " + e.getMessage());
        }
    }
    
    /**
     * 删除工具
     */
    @DeleteMapping("/{id}")
    @Operation(summary = "删除工具", description = "删除指定ID的工具")
    public ApiResponse<Void> deleteTool(@PathVariable String id) {
        try {
            String userId = getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }
            
            toolService.deleteTool(id, userId);
            return ApiResponse.success(null, "删除工具成功");
        } catch (Exception e) {
            log.error("删除工具失败", e);
            return ApiResponse.error(4001, "删除工具失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取工具详情
     */
    @GetMapping("/{id}")
    @Operation(summary = "获取工具详情", description = "获取指定ID的工具详情")
    public ApiResponse<Tool> getTool(@PathVariable String id) {
        try {
            String userId = getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }
            
            Tool tool = toolService.getToolById(id, userId);
            if (tool == null) {
                return ApiResponse.error(4004, "工具不存在");
            }
            
            return ApiResponse.success(tool, "获取工具成功");
        } catch (Exception e) {
            log.error("获取工具失败", e);
            return ApiResponse.error(4001, "获取工具失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取工具列表
     */
    @GetMapping
    @Operation(summary = "获取工具列表", description = "获取当前用户可用工具")
    public ApiResponse<List<Tool>> getTools() {
        try {
            String userId = getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }
            
            List<Tool> tools = toolService.getUserTools(userId);
            return ApiResponse.success(tools, "获取工具列表成功");
        } catch (Exception e) {
            log.error("获取工具列表失败", e);
            return ApiResponse.error(4001, "获取工具列表失败: " + e.getMessage());
        }
    }
}