/**
 * Timeline服务类
 * 整合了SSE管理功能，减少服务层级
 */
export class SimplifiedTimelineService {
  private eventSource: EventSource | null = null;
  private retryCount = 0;
  private maxRetries = 5;
  private retryDelay = 3000;
  private addEventCallback: Function;
  private messageQueue: any[] = [];
  private processingQueue = false;
  

  constructor(addEvent: Function) {
    this.addEventCallback = addEvent;
  }

  /**
   * 建立SSE连接
   */
  connectSSE() {
    // 构造带认证参数的URL
    let eventSourceUrl = '/api/v1/agent/timeline-events';
    
    // 从localStorage获取token
    const token = localStorage.getItem('token');
    
    // 创建一个包含token的URL对象
    const url = new URL(eventSourceUrl, window.location.origin);
    if (token) {
      url.searchParams.append('token', token);
    }
    
    eventSourceUrl = url.toString().replace(window.location.origin, '');
    
    // 如果是相对路径，确保以/开头
    if (!eventSourceUrl.startsWith('/')) {
      eventSourceUrl = '/' + eventSourceUrl;
    }
    
    console.log('[TimelinePanel] 尝试连接SSE:', eventSourceUrl);
    
    // 建立连接
    this.eventSource = new EventSource(eventSourceUrl);
    this.eventSource.onmessage = this.handleMessage.bind(this);
    this.eventSource.onerror = this.handleError.bind(this);
    this.eventSource.onopen = this.handleOpen.bind(this);
    
    return this.eventSource;
  }

  /**
   * 处理SSE消息队列
   */
  private processMessageQueue() {
    // 如果未连接或正在处理队列，则跳过
    if (!this.eventSource || this.eventSource.readyState !== EventSource.OPEN || this.processingQueue || this.messageQueue.length === 0) {
      return;
    }
    
    this.processingQueue = true;
    
    // 批量处理消息以提高性能
    const batchSize = 10;
    const batch = this.messageQueue.splice(0, batchSize);
    
    batch.forEach(data => {
      this.handleSingleSseMessage(data);
    });
    
    this.processingQueue = false;
    
    // 如果还有消息，继续处理
    if (this.messageQueue.length > 0) {
      setTimeout(() => this.processMessageQueue(), 0);
    }
  }
  
  /**
   * 处理单个SSE消息
   */
  private handleSingleSseMessage(data: any) {
    this.addEvent(data);
    
    // 触发embed事件给父组件（如果需要）
    if (data.type === 'embed') {
      window.dispatchEvent(new CustomEvent('embed-event', { detail: data }));
    }
    
    // 重置重试计数
    this.retryCount = 0;
  }
  
  /**
   * 处理消息事件
   */
  private handleMessage(event: MessageEvent) {
    try {
      // 的消息解析
      const data = typeof event.data === 'string' ? JSON.parse(event.data) : event.data;
      if (data) {
        // 将消息加入队列进行批处理，但限制队列大小以避免内存泄漏
        if (this.messageQueue.length < 100) {
          this.messageQueue.push(data);
          this.processMessageQueue();
        } else {
          console.warn('[TimelinePanel] 消息队列已满，丢弃新消息');
        }
      }
    } catch (err) {
      console.error('解析时间轴事件失败:', err);
    }
  }

  /**
   * 处理错误事件
   */
  private handleError(event: Event) {
    console.error('[SSE] 连接错误:', event);
    
    // 尝试重新连接
    if (this.retryCount < this.maxRetries) {
      this.retryCount++;
      setTimeout(() => {
        console.log(`[SSE] 尝试重新连接 (${this.retryCount}/${this.maxRetries})`);
        this.connectSSE();
      }, this.retryDelay * this.retryCount);
      
      this.addEvent({
        type: 'observation',
        title: '重新连接中',
        content: `正在尝试重新连接 (${this.retryCount}/${this.maxRetries})`,
        timestamp: Date.now()
      });
    } else {
      this.addEvent({
        type: 'error',
        title: '连接失败',
        content: '无法连接到服务器事件流，请刷新页面重试',
        timestamp: Date.now()
      });
      
      // 显示用户通知
      // if (typeof window !== 'undefined' && window.alert) {
      //   window.alert('时间轴连接失败，请刷新页面重试');
      // }
    }
  }

  /**
   * 处理连接成功事件
   */
  private handleOpen() {
    console.log('[SSE] 连接已建立');
    
    // 添加连接成功事件到时间轴
    this.addEvent({
      type: 'observation',
      title: 'SSE连接已建立',
      content: '成功连接到服务器事件流',
      timestamp: Date.now()
    });
    
    // 处理队列中积压的消息
    this.processMessageQueue();
  }

  /**
   * 组件卸载时清理资源
   */
  cleanup() {
    if (this.eventSource) {
      this.eventSource.close();
      this.eventSource = null;
    }
  }

  /**
   * 处理来自ChatArea的思考事件
   */
  handleTimelineEvent(e: CustomEvent) {
    const eventData = e.detail;
    console.log('[TimelinePanel] 接收到timeline-event事件:', eventData);
    
    // 确保时间戳存在且有效
    const timestamp = eventData.timestamp || Date.now();

    this.addEvent({
      type: eventData.type || 'thought',
      title: eventData.title || '思考过程',
      content: eventData.content,
      toolName: eventData.toolName,
      toolAction: eventData.toolAction,
      toolInput: eventData.toolInput,
      toolOutput: eventData.toolOutput,
      toolStatus: eventData.toolStatus,
      executionTime: eventData.executionTime,
      embedUrl: eventData.embedUrl,
      embedType: eventData.embedType,
      embedTitle: eventData.embedTitle,
      embedHtmlContent: eventData.embedHtmlContent,
      metadata: eventData.metadata,
      timestamp: timestamp
    });
  }

  /**
   * 处理SSE连接失败事件
   */
  handleSseConnectionFailed() {
    console.error('[SSE] 时间轴事件连接失败，已达到最大重试次数');
    // 添加连接失败事件到时间轴
    this.addEvent({
      type: 'error',
      title: 'SSE连接失败',
      content: '无法连接到服务器事件流，请刷新页面重试',
      timestamp: Date.now()
    });
    
    // 显示用户通知
    // if (typeof window !== 'undefined' && window.alert) {
    //   window.alert('时间轴连接失败，请刷新页面重试');
    // }
  }

  /**
   * 获取当前EventSource实例
   */
  getEventSource(): EventSource | null {
    return this.eventSource;
  }

  /**
   * 添加事件
   */
  addEvent(event: any) {
    if (this.addEventCallback) {
      this.addEventCallback(event);
    }
  }

  /**
   * 清除Timeline
   */
  clearTimeline() {
    // 实现清除逻辑
  }

  /**
   * 获取事件类型标签
   */
  getEventTypeLabel(eventTypeLabels: Record<string, string>, type: string): string {
    return eventTypeLabels[type] || type;
  }

  /**
   * 格式化时间
   */
  formatTime(timestamp: number): string {
    const date = new Date(timestamp);
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    const seconds = String(date.getSeconds()).padStart(2, '0');
    return `${hours}:${minutes}:${seconds}`;
  }

  /**
   * 获取事件的展开状态
   */
  getExpandedState(): boolean {
    // 实现
    return false;
  }

  /**
   * 切换事件详细信息的展开状态
   */
  toggleExpand() {
    // 实现
  }

  /**
   * 工具事件类型判断
   */
  isToolEventType(type: string): boolean {
    return ['tool_call', 'tool_result', 'tool_error'].includes(type);
  }

  /**
   * 工具输入有效性检查
   */
  hasValidToolInput(event: any): boolean {
    return event.type === 'tool_call' && event.toolInput !== null && event.toolInput !== undefined;
  }

  /**
   * 工具输出有效性检查
   */
  hasValidToolOutput(event: any): boolean {
    return event.type === 'tool_result' && event.toolOutput !== null && event.toolOutput !== undefined;
  }
}