package pangea.hiagent.security;

import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.common.utils.JwtUtil;

import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

/**
 * JWT认证过滤器
 * 从请求头中提取JWT Token并进行验证
 */
@Slf4j
@Component
public class JwtAuthenticationFilter extends OncePerRequestFilter {
    
    private final JwtUtil jwtUtil;
    
    public JwtAuthenticationFilter(JwtUtil jwtUtil) {
        this.jwtUtil = jwtUtil;
    }
    
    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain)
            throws ServletException, IOException {
        boolean isStreamEndpoint = request.getRequestURI().contains("/api/v1/agent/chat-stream");
        boolean isTimelineEndpoint = request.getRequestURI().contains("/api/v1/agent/timeline-events");
        
        if (isStreamEndpoint) {
            log.info("处理Agent流式对话请求: {} {}", request.getMethod(), request.getRequestURI());
        }
        if (isTimelineEndpoint) {
            log.info("处理时间轴事件订阅请求: {} {}", request.getMethod(), request.getRequestURI());
        }
        
        // 对于OPTIONS请求，直接放行
        if ("OPTIONS".equalsIgnoreCase(request.getMethod())) {
            log.debug("OPTIONS请求，直接放行");
            filterChain.doFilter(request, response);
            return;
        }
        
        try {
            String token = extractTokenFromRequest(request);
            
            log.debug("JWT过滤器处理请求: {} {}，提取到token: {}", request.getMethod(), request.getRequestURI(), token);
            
            if (StringUtils.hasText(token)) {
                log.debug("开始JWT验证，token长度: {}", token.length());
                // 验证token是否有效
                boolean isValid = jwtUtil.validateToken(token);
                log.debug("JWT验证结果: {}", isValid);
                if (isValid) {
                    String userId = jwtUtil.getUserIdFromToken(token);
                    log.debug("JWT验证通过，用户ID: {}", userId);
                    
                    if (userId != null) {
                        // 创建认证对象，添加基本权限
                        List<SimpleGrantedAuthority> authorities = Collections.singletonList(new SimpleGrantedAuthority("ROLE_USER"));
                        UsernamePasswordAuthenticationToken authentication = 
                                new UsernamePasswordAuthenticationToken(userId, null, authorities);
                        SecurityContextHolder.getContext().setAuthentication(authentication);
                        log.debug("已设置SecurityContext中的认证信息，用户ID: {}, 权限: {}", userId, authentication.getAuthorities());
                    } else {
                        log.warn("从token中提取的用户ID为空");
                    }
                } else {
                    log.warn("JWT验证失败，token可能已过期或无效");
                    // 检查token是否过期
                    boolean isExpired = jwtUtil.isTokenExpired(token);
                    log.warn("Token过期状态: {}", isExpired);
                }
            } else {
                log.debug("未找到有效的token");
                // 记录请求信息以便调试
                log.debug("请求URL: {}", request.getRequestURL());
                log.debug("请求方法: {}", request.getMethod());
                log.debug("Authorization头: {}", request.getHeader("Authorization"));
                log.debug("token参数: {}", request.getParameter("token"));
            }
        } catch (Exception e) {
            log.error("JWT认证处理异常", e);
            // 不在此处发送错误响应，让Spring Security的ExceptionTranslationFilter处理
            // 这样可以避免响应被提前提交
        }        
        // 特别处理流式端点的权限问题
        if (isStreamEndpoint || isTimelineEndpoint) {
            // 检查是否已认证
            if (SecurityContextHolder.getContext().getAuthentication() == null) {
                log.warn("流式端点未认证访问: {} {}", request.getMethod(), request.getRequestURI());
                // 对于SSE端点，如果未认证，我们不立即返回错误，而是让后续处理决定
                // 因为客户端可能会在重新连接时带上token
            }
            
            // 对于SSE端点，直接执行过滤器链，不进行额外的响应检查
            filterChain.doFilter(request, response);
            log.debug("JwtAuthenticationFilter处理完成(SSE端点): {} {}", request.getMethod(), request.getRequestURI());
            return;
        }
        
        // 继续执行过滤器链，让Spring Security的其他过滤器处理认证和授权
        // 这样可以让ExceptionTranslationFilter和AuthorizationFilter正确处理认证失败和权限拒绝
        filterChain.doFilter(request, response);
        
        log.debug("JwtAuthenticationFilter处理完成: {} {}", request.getMethod(), request.getRequestURI());
    }
    
    /**
     * 从请求头或参数中提取Token
     * 为了支持SSE连接，我们还需要检查URL参数中的token
     */
    private String extractTokenFromRequest(HttpServletRequest request) {
        // 首先尝试从请求头中提取Token
        String authHeader = request.getHeader("Authorization");
        log.debug("从请求头中提取Authorization: {}", authHeader);
        if (StringUtils.hasText(authHeader) && authHeader.startsWith("Bearer ")) {
            String token = authHeader.substring(7);
            log.debug("从Authorization头中提取到token");
            return token;
        }
        
        // 如果请求头中没有Token，则尝试从URL参数中提取
        // 这对于SSE连接特别有用，因为浏览器在自动重连时可能不会发送Authorization头
        String tokenParam = request.getParameter("token");
        log.debug("从URL参数中提取token参数: {}", tokenParam);
        if (StringUtils.hasText(tokenParam)) {
            log.debug("从URL参数中提取到token");
            return tokenParam;
        }
        
        log.debug("未找到有效的token");
        return null;
    }
}