package pangea.hiagent.llm;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.LlmConfig;

import jakarta.annotation.PostConstruct;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * 模型适配器管理器
 * 负责管理和选择合适的模型适配器
 */
@Slf4j
@Component
public class ModelAdapterManager {
    
    @Autowired
    private List<ModelAdapter> adapters;
    
    private Map<String, ModelAdapter> adapterMap;
    
    @PostConstruct
    public void init() {
        adapterMap = new HashMap<>();
        for (ModelAdapter adapter : adapters) {
            adapterMap.put(adapter.getProviderName().toLowerCase(), adapter);
            log.info("注册模型适配器: {}", adapter.getProviderName());
        }
        log.info("模型适配器初始化完成，共注册 {} 个适配器", adapterMap.size());
    }
    
    /**
     * 根据配置创建ChatModel实例
     * 
     * @param config LLM配置
     * @return ChatModel实例
     * @throws IllegalArgumentException 当配置无效或不支持时
     * @throws RuntimeException 当创建模型失败时
     */
    public ChatModel createChatModel(LlmConfig config) {
        log.debug("开始创建ChatModel，配置信息: {}", config);
        
        // 验证配置
        if (config == null) {
            log.warn("LLM配置为空");
            throw new IllegalArgumentException("LLM配置不能为空");
        }
        
        if (!config.getEnabled()) {
            log.warn("LLM配置未启用: {}", config.getId());
            throw new IllegalArgumentException("LLM配置未启用: " + config.getId());
        }
        
        String provider = config.getProvider();
        if (provider == null || provider.isEmpty()) {
            log.warn("LLM配置缺少提供商信息");
            throw new IllegalArgumentException("LLM配置缺少提供商信息");
        }
        
        // 查找适配器
        ModelAdapter adapter = adapterMap.get(provider.toLowerCase());
        if (adapter == null) {
            log.warn("不支持的LLM提供商: {}", provider);
            throw new IllegalArgumentException("不支持的LLM提供商: " + provider);
        }
        
        log.debug("找到适配器: {}", adapter.getClass().getSimpleName());
        
        // 验证配置有效性
        if (!adapter.validateConfig(config)) {
            log.warn("LLM配置验证失败: {}", provider);
            throw new IllegalArgumentException("LLM配置验证失败: " + provider);
        }
        
        // 创建模型
        try {
            log.debug("调用适配器创建ChatModel");
            ChatModel model = adapter.createChatModel(config);
            log.info("成功创建ChatModel，提供商: {}", provider);
            return model;
        } catch (Exception e) {
            log.error("创建ChatModel失败，提供商: {}, 错误: {}", provider, e.getMessage(), e);
            throw new RuntimeException("创建ChatModel失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 获取指定提供商的适配器
     * 
     * @param provider 提供商名称
     * @return 模型适配器
     */
    public Optional<ModelAdapter> getAdapter(String provider) {
        if (provider == null || provider.isEmpty()) {
            return Optional.empty();
        }
        return Optional.ofNullable(adapterMap.get(provider.toLowerCase()));
    }
    
    /**
     * 获取所有已注册的适配器
     * 
     * @return 适配器映射的不可变视图
     */
    public Map<String, ModelAdapter> getAdapters() {
        return Collections.unmodifiableMap(adapterMap);
    }
}