package pangea.hiagent.tools;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import org.springframework.ai.tool.annotation.Tool;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

/**
 * 文件处理工具类
 * 提供文件读写和管理功能，支持多种文本格式文件
 */
@Slf4j
@Component
public class FileProcessingTools {
    
    // 支持的文本文件扩展名
    private static final List<String> TEXT_FILE_EXTENSIONS = Arrays.asList(
        ".txt", ".md", ".java", ".html", ".htm", ".css", ".js", ".json", 
        ".xml", ".yaml", ".yml", ".properties", ".sql", ".py", ".cpp", 
        ".c", ".h", ".cs", ".php", ".rb", ".go", ".rs", ".swift", ".kt", 
        ".scala", ".sh", ".bat", ".cmd", ".ps1", ".log", ".csv", ".ts", 
        ".jsx", ".tsx", ".vue", ".scss", ".sass", ".less"
    );
    
    // 支持的图片文件扩展名
    private static final List<String> IMAGE_FILE_EXTENSIONS = Arrays.asList(
        ".jpg", ".jpeg", ".png", ".gif", ".bmp", ".svg", ".webp", ".ico"
    );
    
    // 默认文件存储目录
    private static final String DEFAULT_STORAGE_DIR = "storage";
    
    /**
     * 检查文件是否为文本文件
     * @param filePath 文件路径
     * @return 是否为文本文件
     */
    private boolean isTextFile(String filePath) {
        if (filePath == null || filePath.isEmpty()) {
            return false;
        }
        
        String lowerPath = filePath.toLowerCase();
        return TEXT_FILE_EXTENSIONS.stream().anyMatch(lowerPath::endsWith);
    }
    
    /**
     * 检查文件是否为图片文件
     * @param filePath 文件路径
     * @return 是否为图片文件
     */
    private boolean isImageFile(String filePath) {
        if (filePath == null || filePath.isEmpty()) {
            return false;
        }
        
        String lowerPath = filePath.toLowerCase();
        return IMAGE_FILE_EXTENSIONS.stream().anyMatch(lowerPath::endsWith);
    }
    
    /**
     * 检查文件是否为支持的文件类型（文本或图片）
     * @param filePath 文件路径
     * @return 是否为支持的文件类型
     */
    private boolean isSupportedFile(String filePath) {
        return isTextFile(filePath) || isImageFile(filePath);
    }
    
    /**
     * 处理文件路径，支持默认相对路径和随机文件名
     * @param filePath 文件路径，如果为null或空则生成随机文件名
     * @param extension 文件扩展名
     * @return 处理后的完整文件路径
     */
    private String processFilePath(String filePath, String extension) {
        // 如果filePath为null或空，则生成随机文件名
        if (filePath == null || filePath.isEmpty()) {
            // 确保默认存储目录存在
            File storageDir = new File(DEFAULT_STORAGE_DIR);
            if (!storageDir.exists()) {
                storageDir.mkdirs();
            }
            
            // 生成随机文件名
            String randomFileName = UUID.randomUUID().toString().replace("-", "") + extension;
            filePath = DEFAULT_STORAGE_DIR + File.separator + randomFileName;
            log.debug("生成随机文件名: {}", filePath);
        } else {
            // 处理相对路径
            File file = new File(filePath);
            if (!file.isAbsolute()) {
                // 如果是相对路径，转换为相对于当前工作目录的绝对路径
                filePath = file.getAbsolutePath();
                log.debug("转换相对路径为绝对路径: {}", filePath);
            }
        }
        
        return filePath;
    }
    
    /**
     * 根据内容自动推断文件扩展名
     * @param content 文件内容
     * @return 推断的文件扩展名
     */
    private String inferFileExtension(String content) {
        if (content == null || content.isEmpty()) {
            return ".txt";
        }
        
        // 简单的内容分析来推断文件类型
        String trimmedContent = content.trim();
        if (trimmedContent.startsWith("{") || trimmedContent.startsWith("[")) {
            return ".json";
        } else if (trimmedContent.startsWith("<!DOCTYPE html") || trimmedContent.startsWith("<html")) {
            return ".html";
        } else if (trimmedContent.startsWith("#") || trimmedContent.startsWith("##")) {
            return ".md";
        } else if (trimmedContent.contains("public class") || trimmedContent.contains("public static void main")) {
            return ".java";
        } else if (trimmedContent.startsWith("<?xml")) {
            return ".xml";
        } else {
            return ".txt";
        }
    }
    
    @Tool(description = "读取文本文件内容，支持多种文本格式")
    public String readFile(String filePath) {
        return readFileWithEncoding(filePath, "UTF-8");
    }
    
    @Tool(description = "读取文本文件内容，支持指定字符编码")
    public String readFileWithEncoding(String filePath, String encoding) {
        log.debug("开始读取文件: {}, 编码: {}", filePath, encoding);
        try {
            if (filePath == null || filePath.isEmpty()) {
                log.warn("文件路径不能为空");
                return "错误：文件路径不能为空";
            }
            
            // 处理相对路径
            File file = new File(filePath);
            if (!file.isAbsolute()) {
                // 如果是相对路径，转换为相对于当前工作目录的绝对路径
                filePath = file.getAbsolutePath();
                log.debug("转换相对路径为绝对路径: {}", filePath);
            }
            
            // 检查文件是否存在
            if (!file.exists()) {
                log.warn("文件不存在: {}", filePath);
                return "错误：文件不存在";
            }
            
            // 检查是否为支持的文件类型
            if (!isSupportedFile(filePath)) {
                log.warn("文件不是支持的格式: {}", filePath);
                return "错误：文件不是支持的格式";
            }
            
            // 确定字符编码
            Charset charset = StandardCharsets.UTF_8;
            if (encoding != null && !encoding.isEmpty()) {
                try {
                    charset = Charset.forName(encoding);
                } catch (Exception e) {
                    log.warn("无效的字符编码，使用默认UTF-8: {}", encoding);
                }
            }
            
            // 读取文件内容
            String content = new String(Files.readAllBytes(Paths.get(filePath)), charset);
            log.debug("文件读取成功，文件大小: {} 字节，字符数: {}", content.getBytes(charset).length, content.length());
            return content;
        } catch (IOException e) {
            log.error("读取文件时发生IO错误: {}", filePath, e);
            return "读取文件时发生错误：" + e.getMessage();
        } catch (Exception e) {
            log.error("处理文件时发生未知错误: {}", filePath, e);
            return "处理文件时发生未知错误：" + e.getMessage();
        }
    }
    
    @Tool(description = "写入内容到文本文件，支持多种文本格式，支持默认相对路径和随机文件名")
    public String writeFile(String filePath, String content) {
        return writeFileWithEncoding(filePath, content, "UTF-8", false);
    }
    
    @Tool(description = "写入内容到文本文件，支持指定字符编码和追加模式，支持默认相对路径和随机文件名")
    public String writeFileWithEncoding(String filePath, String content, String encoding, boolean append) {
        log.debug("开始写入文件: {}, 编码: {}, 追加模式: {}", filePath, encoding, append);
        try {
            // 处理文件路径，支持默认相对路径和随机文件名
            String extension = inferFileExtension(content);
            String processedFilePath = processFilePath(filePath, extension);
            
            // 检查是否为支持的文件类型
            if (!isSupportedFile(processedFilePath)) {
                log.warn("文件不是支持的格式: {}", processedFilePath);
                return "错误：文件不是支持的格式";
            }
            
            // 确保父目录存在
            File file = new File(processedFilePath);
            File parentDir = file.getParentFile();
            if (parentDir != null && !parentDir.exists()) {
                if (!parentDir.mkdirs()) {
                    log.warn("无法创建目录: {}", parentDir.getAbsolutePath());
                    return "错误：无法创建目录";
                }
            }
            
            // 确定字符编码
            Charset charset = StandardCharsets.UTF_8;
            if (encoding != null && !encoding.isEmpty()) {
                try {
                    charset = Charset.forName(encoding);
                } catch (Exception e) {
                    log.warn("无效的字符编码，使用默认UTF-8: {}", encoding);
                }
            }
            
            // 处理空内容
            if (content == null) {
                content = "";
            }
            
            // 写入文件
            if (append) {
                Files.write(Paths.get(processedFilePath), content.getBytes(charset), 
                           StandardOpenOption.CREATE, StandardOpenOption.APPEND);
            } else {
                Files.write(Paths.get(processedFilePath), content.getBytes(charset));
            }
            
            log.debug("文件写入成功，内容大小: {} 字符，文件路径: {}", content.length(), processedFilePath);
            return "文件写入成功，文件路径: " + processedFilePath;
        } catch (IOException e) {
            log.error("写入文件时发生IO错误: {}", filePath, e);
            return "写入文件时发生错误：" + e.getMessage();
        } catch (Exception e) {
            log.error("处理文件时发生未知错误: {}", filePath, e);
            return "处理文件时发生未知错误：" + e.getMessage();
        }
    }
    
    @Tool(description = "追加内容到文本文件末尾，支持默认相对路径和随机文件名")
    public String appendToFile(String filePath, String content) {
        return writeFileWithEncoding(filePath, content, "UTF-8", true);
    }
    
    @Tool(description = "获取文件大小")
    public String getFileSize(String filePath) {
        log.debug("获取文件大小: {}", filePath);
        try {
            if (filePath == null || filePath.isEmpty()) {
                log.warn("文件路径不能为空");
                return "错误：文件路径不能为空";
            }
            
            // 处理相对路径
            File file = new File(filePath);
            if (!file.isAbsolute()) {
                // 如果是相对路径，转换为相对于当前工作目录的绝对路径
                filePath = file.getAbsolutePath();
                log.debug("转换相对路径为绝对路径: {}", filePath);
            }
            
            if (!file.exists()) {
                log.warn("文件不存在: {}", filePath);
                return "错误：文件不存在";
            }
            
            long size = file.length();
            log.debug("文件大小: {} 字节", size);
            return "文件大小：" + size + " 字节";
        } catch (Exception e) {
            log.error("获取文件大小时发生错误: {}", filePath, e);
            return "获取文件大小时发生错误：" + e.getMessage();
        }
    }
    
    @Tool(description = "检查文件是否存在")
    public boolean fileExists(String filePath) {
        log.debug("检查文件是否存在: {}", filePath);
        if (filePath == null || filePath.isEmpty()) {
            log.warn("文件路径不能为空");
            return false;
        }
        
        // 处理相对路径
        File file = new File(filePath);
        if (!file.isAbsolute()) {
            // 如果是相对路径，转换为相对于当前工作目录的绝对路径
            filePath = file.getAbsolutePath();
            log.debug("转换相对路径为绝对路径: {}", filePath);
        }
        
        file = new File(filePath);
        boolean exists = file.exists();
        log.debug("文件存在状态: {}", exists);
        return exists;
    }
    
    @Tool(description = "获取文件信息，包括大小、是否为文本文件等")
    public String getFileInfo(String filePath) {
        log.debug("获取文件信息: {}", filePath);
        try {
            if (filePath == null || filePath.isEmpty()) {
                log.warn("文件路径不能为空");
                return "错误：文件路径不能为空";
            }
            
            // 处理相对路径
            File file = new File(filePath);
            if (!file.isAbsolute()) {
                // 如果是相对路径，转换为相对于当前工作目录的绝对路径
                filePath = file.getAbsolutePath();
                log.debug("转换相对路径为绝对路径: {}", filePath);
            }
            
            file = new File(filePath);
            if (!file.exists()) {
                log.warn("文件不存在: {}", filePath);
                return "错误：文件不存在";
            }
            
            StringBuilder info = new StringBuilder();
            info.append("文件路径: ").append(filePath).append("\n");
            info.append("文件大小: ").append(file.length()).append(" 字节\n");
            info.append("是否为文本文件: ").append(isTextFile(filePath)).append("\n");
            info.append("是否为图片文件: ").append(isImageFile(filePath)).append("\n");
            info.append("是否为支持的文件类型: ").append(isSupportedFile(filePath)).append("\n");
            info.append("最后修改时间: ").append(new java.util.Date(file.lastModified())).append("\n");
            
            log.debug("文件信息获取成功: {}", filePath);
            return info.toString();
        } catch (Exception e) {
            log.error("获取文件信息时发生错误: {}", filePath, e);
            return "获取文件信息时发生错误：" + e.getMessage();
        }
    }
    
    @Tool(description = "生成随机文件名并返回完整路径")
    public String generateRandomFileName(String extension) {
        log.debug("生成随机文件名，扩展名: {}", extension);
        try {
            // 确保默认存储目录存在
            File storageDir = new File(DEFAULT_STORAGE_DIR);
            if (!storageDir.exists()) {
                storageDir.mkdirs();
            }
            
            // 处理扩展名
            if (extension == null || extension.isEmpty()) {
                extension = ".txt";
            } else if (!extension.startsWith(".")) {
                extension = "." + extension;
            }
            
            // 生成随机文件名
            String randomFileName = UUID.randomUUID().toString().replace("-", "") + extension;
            String fullPath = DEFAULT_STORAGE_DIR + File.separator + randomFileName;
            
            log.debug("生成随机文件名: {}", fullPath);
            return fullPath;
        } catch (Exception e) {
            log.error("生成随机文件名时发生错误", e);
            return "生成随机文件名时发生错误：" + e.getMessage();
        }
    }
}