package pangea.hiagent.memory;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.memory.ChatMemory;
import org.springframework.ai.chat.messages.AssistantMessage;
import org.springframework.ai.chat.messages.UserMessage;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Service;
import pangea.hiagent.model.Agent;

import java.util.Collections;
import java.util.List;

/**
 * 内存服务类
 * 负责管理聊天内存和会话相关功能
 */
@Slf4j
@Service
public class MemoryService {
    
    @Autowired
    private ChatMemory chatMemory;
    
    @Autowired
    private SmartHistorySummarizer smartHistorySummarizer;
    
    /**
     * 为每个用户-Agent组合创建唯一的会话ID
     * @param agent Agent对象
     * @return 会话ID
     */
    public String generateSessionId(Agent agent) {
        return generateSessionId(agent, null);
    }
    
    /**
     * 为每个用户-Agent组合创建唯一的会话ID
     * @param agent Agent对象
     * @param userId 用户ID（可选，如果未提供则从SecurityContext获取）
     * @return 会话ID
     */
    public String generateSessionId(Agent agent, String userId) {
        if (userId == null) {
            userId = getCurrentUserId();
        }
        return userId + "_" + agent.getId();
    }
    
    /**
     * 获取当前认证用户ID
     * @return 用户ID
     */
    private String getCurrentUserId() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        return (authentication != null && authentication.getPrincipal() != null) ? 
            (String) authentication.getPrincipal() : null;
    }
    
    /**
     * 添加用户消息到ChatMemory
     * @param sessionId 会话ID
     * @param userMessage 用户消息
     */
    public void addUserMessageToMemory(String sessionId, String userMessage) {
        UserMessage userMsg = new UserMessage(userMessage);
        chatMemory.add(sessionId, Collections.singletonList(userMsg));
    }
    
    /**
     * 添加助手回复到ChatMemory
     * @param sessionId 会话ID
     * @param assistantMessage 助手回复
     */
    public void addAssistantMessageToMemory(String sessionId, String assistantMessage) {
        AssistantMessage assistantMsg = new AssistantMessage(assistantMessage);
        chatMemory.add(sessionId, Collections.singletonList(assistantMsg));
    }
    
    /**
     * 获取历史消息
     * @param sessionId 会话ID
     * @param historyLength 历史记录长度
     * @return 历史消息列表
     */
    public List<org.springframework.ai.chat.messages.Message> getHistoryMessages(String sessionId, int historyLength) {
        return chatMemory.get(sessionId, historyLength);
    }
    
    /**
     * 获取智能摘要后的历史消息
     * @param sessionId 会话ID
     * @param historyLength 历史记录长度
     * @return 智能摘要后的历史消息列表
     */
    public List<org.springframework.ai.chat.messages.Message> getSmartHistoryMessages(String sessionId, int historyLength) {
        List<org.springframework.ai.chat.messages.Message> historyMessages = chatMemory.get(sessionId, historyLength);
        return smartHistorySummarizer.summarize(historyMessages, historyLength);
    }
}