package pangea.hiagent.security;

import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.PermissionEvaluator;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.TimerConfig;
import pangea.hiagent.service.AgentService;
import pangea.hiagent.service.TimerService;

import java.io.Serializable;

/**
 * 自定义权限评估器
 * 用于实现细粒度的资源级权限控制
 */
@Slf4j
@Component("permissionEvaluator")
public class DefaultPermissionEvaluator implements PermissionEvaluator {
    
    private final AgentService agentService;
    private final TimerService timerService;
    
    public DefaultPermissionEvaluator(AgentService agentService, TimerService timerService) {
        this.agentService = agentService;
        this.timerService = timerService;
    }
    
    /**
     * 检查用户是否有权访问指定资源
     */
    @Override
    public boolean hasPermission(Authentication authentication, Object targetDomainObject, Object permission) {
        if (authentication == null || targetDomainObject == null || !(permission instanceof String)) {
            return false;
        }
        
        String userId = (String) authentication.getPrincipal();
        String perm = (String) permission;
        
        // 处理Agent访问权限
        if (targetDomainObject instanceof Agent) {
            Agent agent = (Agent) targetDomainObject;
            return checkAgentAccess(userId, agent, perm);
        } 
        // 处理TimerConfig访问权限
        else if (targetDomainObject instanceof TimerConfig) {
            TimerConfig timer = (TimerConfig) targetDomainObject;
            return checkTimerAccess(userId, timer, perm);
        } 
        // 处理基于ID的资源访问
        else if (targetDomainObject instanceof String) {
            // 这种情况在hasPermission(Authentication, Serializable, String, Object)方法中处理
            return false;
        }
        
        return false;
    }
    
    @Override
    public boolean hasPermission(Authentication authentication, Serializable targetId, String targetType, Object permission) {
        if (authentication == null || targetId == null || targetType == null || !(permission instanceof String)) {
            return false;
        }
        
        String userId = (String) authentication.getPrincipal();
        String perm = (String) permission;
        
        // 处理基于ID的权限检查
        if ("Agent".equals(targetType)) {
            Agent agent = agentService.getAgent(targetId.toString());
            if (agent == null) {
                return false;
            }
            return checkAgentAccess(userId, agent, perm);
        } 
        // 处理TimerConfig资源的权限检查
        else if ("TimerConfig".equals(targetType)) {
            TimerConfig timer = timerService.getTimerById(targetId.toString());
            if (timer == null) {
                return false;
            }
            return checkTimerAccess(userId, timer, perm);
        }
        
        return false;
    }
    
    /**
     * 检查用户对Agent的访问权限
     */
    private boolean checkAgentAccess(String userId, Agent agent, String permission) {
        // 管理员可以访问所有Agent
        if (isAdminUser(userId)) {
            return true;
        }
        
        // 检查Agent所有者
        if (agent.getOwner().equals(userId)) {
            return true;
        }
        
        // 根据权限类型进行检查
        switch (permission.toLowerCase()) {
            case "read":
                // 所有用户都可以读取公开的Agent（如果有此概念）
                return false; // 暂时不支持公开Agent
            case "write":
            case "delete":
            case "execute":
                // 只有所有者可以写入、删除或执行Agent
                return agent.getOwner().equals(userId);
            default:
                return false;
        }
    }
    
    /**
     * 检查用户对TimerConfig的访问权限
     */
    private boolean checkTimerAccess(String userId, TimerConfig timer, String permission) {
        // 管理员可以访问所有定时器
        if (isAdminUser(userId)) {
            return true;
        }
        
        // 检查定时器创建者
        if (timer.getCreatedBy() != null && timer.getCreatedBy().equals(userId)) {
            return true;
        }
        
        // 根据权限类型进行检查
        switch (permission.toLowerCase()) {
            case "read":
                // 所有用户都可以读取公开的定时器（如果有此概念）
                return false; // 暂时不支持公开定时器
            case "write":
            case "delete":
                // 只有创建者可以修改或删除定时器
                return timer.getCreatedBy() != null && timer.getCreatedBy().equals(userId);
            default:
                return false;
        }
    }
    
    /**
     * 检查是否为管理员用户
     */
    private boolean isAdminUser(String userId) {
        // 这里可以根据实际需求实现管理员检查逻辑
        // 例如查询数据库或检查特殊用户ID
        // 当前实现保留原有逻辑，但可以通过配置或数据库来管理管理员用户
        return "admin".equals(userId) || "user-001".equals(userId);
    }
}