package pangea.hiagent.tools.processor;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;
import pangea.hiagent.service.ToolConfigService;
import pangea.hiagent.tools.annotation.ToolParam;

import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.List;

/**
 * 工具参数处理器
 * 用于处理工具类中的@ToolParam注解，将数据库中的参数值注入到工具类字段
 */
@Slf4j
@Component
@Scope(ConfigurableBeanFactory.SCOPE_SINGLETON)
public class ToolParamProcessor implements BeanPostProcessor {
    
    private final ToolConfigService toolConfigService;
    
    // 构造函数注入
    public ToolParamProcessor(ToolConfigService toolConfigService) {
        this.toolConfigService = toolConfigService;
    }
    
    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        // 检查Bean是否为工具类（位于tools包下，且带有@Component注解）
        Class<?> beanClass = bean.getClass();
        String packageName = beanClass.getPackage().getName();
        
        if (packageName.contains("pangea.hiagent.tools") && beanClass.isAnnotationPresent(Component.class)) {
            log.debug("处理工具类参数，Bean名称：{}", beanName);
            injectParams(bean);
        }
        
        return bean;
    }
    
    /**
     * 注入参数值到工具类字段
     * @param bean 工具类实例
     */
    private void injectParams(Object bean) {
        Class<?> beanClass = bean.getClass();
        String toolName = beanClass.getSimpleName();
        
        // 获取所有字段，包括父类字段
        List<Field> fields = getAllFields(beanClass);
        
        for (Field field : fields) {
            if (field.isAnnotationPresent(ToolParam.class)) {
                ToolParam annotation = field.getAnnotation(ToolParam.class);
                String paramName = annotation.name().isEmpty() ? field.getName() : annotation.name();
                
                // 从数据库获取参数值，如果不存在则使用默认值
                String paramValue = toolConfigService.getParamValue(toolName, paramName);
                if (paramValue == null) {
                    paramValue = annotation.defaultValue();
                    log.debug("参数值不存在，使用默认值，工具名称：{}，参数名称：{}，默认值：{}", 
                            toolName, paramName, paramValue);
                }
                
                // 设置字段值
                field.setAccessible(true);
                try {
                    // 根据字段类型转换参数值
                    injectFieldValue(bean, field, paramValue);
                    log.debug("参数值注入成功，工具名称：{}，参数名称：{}，字段类型：{}，值：{}", 
                            toolName, paramName, field.getType().getName(), paramValue);
                } catch (Exception e) {
                    log.error("参数值注入失败，工具名称：{}，参数名称：{}，字段类型：{}，值：{}", 
                            toolName, paramName, field.getType().getName(), paramValue, e);
                }
            }
        }
    }
    
    /**
     * 递归获取所有字段，包括父类字段
     * @param clazz 类对象
     * @return 字段列表
     */
    private List<Field> getAllFields(Class<?> clazz) {
        List<Field> fields = Arrays.asList(clazz.getDeclaredFields());
        
        Class<?> superClass = clazz.getSuperclass();
        if (superClass != null && !superClass.equals(Object.class)) {
            fields.addAll(getAllFields(superClass));
        }
        
        return fields;
    }
    
    /**
     * 根据字段类型注入参数值
     * @param bean 工具类实例
     * @param field 字段对象
     * @param paramValue 参数值字符串
     * @throws IllegalAccessException 访问权限异常
     */
    private void injectFieldValue(Object bean, Field field, String paramValue) throws IllegalAccessException {
        Class<?> fieldType = field.getType();
        
        if (fieldType == String.class) {
            field.set(bean, paramValue);
        } else if (fieldType == int.class || fieldType == Integer.class) {
            field.set(bean, Integer.parseInt(paramValue));
        } else if (fieldType == long.class || fieldType == Long.class) {
            field.set(bean, Long.parseLong(paramValue));
        } else if (fieldType == boolean.class || fieldType == Boolean.class) {
            field.set(bean, Boolean.parseBoolean(paramValue));
        } else if (fieldType == double.class || fieldType == Double.class) {
            field.set(bean, Double.parseDouble(paramValue));
        } else if (fieldType == float.class || fieldType == Float.class) {
            field.set(bean, Float.parseFloat(paramValue));
        } else if (fieldType == short.class || fieldType == Short.class) {
            field.set(bean, Short.parseShort(paramValue));
        } else if (fieldType == byte.class || fieldType == Byte.class) {
            field.set(bean, Byte.parseByte(paramValue));
        } else if (fieldType == char.class || fieldType == Character.class) {
            field.set(bean, paramValue.charAt(0));
        } else {
            // 对于其他类型，直接设置为null
            field.set(bean, null);
            log.warn("不支持的字段类型，工具名称：{}，参数名称：{}，字段类型：{}", 
                    bean.getClass().getSimpleName(), field.getName(), fieldType.getName());
        }
    }
}