package pangea.hiagent.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import pangea.hiagent.dto.ApiResponse;
import pangea.hiagent.dto.TimerExecutionHistoryDto;
import pangea.hiagent.service.HistoryService;

/**
 * 定时器执行历史API控制器
 * 负责处理执行历史的查询和管理
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/timer-history")
public class TimerHistoryController {

    private final HistoryService historyService;

    public TimerHistoryController(HistoryService historyService) {
        this.historyService = historyService;
    }

    /**
     * 获取执行历史列表
     */
    @GetMapping
    public ApiResponse<Page<TimerExecutionHistoryDto>> listExecutionHistory(
            @RequestParam(required = false) String timerId,
            @RequestParam(required = false) Integer success,
            @RequestParam(required = false) String startTime,
            @RequestParam(required = false) String endTime,
            @RequestParam(defaultValue = "1") int page,
            @RequestParam(defaultValue = "10") int size) {
        try {
            log.info("获取执行历史列表，timerId: {}, success: {}, startTime: {}, endTime: {}", 
                    timerId, success, startTime, endTime);
            
            Page<TimerExecutionHistoryDto> historyPage = historyService.getExecutionHistoryList(
                    timerId, success, startTime, endTime, page, size);
            
            return ApiResponse.success(historyPage, "获取执行历史成功");
        } catch (Exception e) {
            log.error("获取执行历史失败", e);
            return ApiResponse.error(4001, "获取执行历史失败: " + e.getMessage());
        }
    }

    /**
     * 获取指定定时器的执行历史
     */
    @GetMapping("/{timerId}")
    public ApiResponse<Page<TimerExecutionHistoryDto>> listTimerExecutionHistory(
            @PathVariable String timerId,
            @RequestParam(defaultValue = "1") int page,
            @RequestParam(defaultValue = "10") int size) {
        try {
            log.info("获取定时器 {} 的执行历史", timerId);
            
            Page<TimerExecutionHistoryDto> historyPage = historyService.getExecutionHistoryByTimerId(
                    timerId, page, size);
            
            return ApiResponse.success(historyPage, "获取定时器执行历史成功");
        } catch (Exception e) {
            log.error("获取定时器执行历史失败", e);
            return ApiResponse.error(4001, "获取定时器执行历史失败: " + e.getMessage());
        }
    }

    /**
     * 获取执行历史详情
     */
    @GetMapping("/detail/{id}")
    public ApiResponse<TimerExecutionHistoryDto> getExecutionHistoryDetail(@PathVariable Long id) {
        try {
            log.info("获取执行历史详情: {}", id);
            
            TimerExecutionHistoryDto historyDetail = historyService.getExecutionHistoryDetail(id);
            if (historyDetail == null) {
                return ApiResponse.error(4004, "执行历史不存在");
            }
            
            return ApiResponse.success(historyDetail, "获取执行历史详情成功");
        } catch (Exception e) {
            log.error("获取执行历史详情失败", e);
            return ApiResponse.error(4001, "获取执行历史详情失败: " + e.getMessage());
        }
    }
}