// WebSocket服务
import { addLog } from '@/utils/logUtils';
import { BinaryFragmentBuffer, handleBinaryMessage } from './binaryMessageHandler';
import { TokenUtils } from '@/utils/tokenUtils';

interface WebSocketServiceOptions {
  onMessage?: (data: any) => void;
  onOpen?: () => void;
  onClose?: (event: CloseEvent) => void;
  onError?: (error: any) => void;
}

export class WebSocketService {
  private ws: WebSocket | null = null;
  private url: string = '';
  private reconnectAttempts: number = 0;
  private maxReconnectAttempts: number = 5;
  private reconnectDelay: number = 3000;
  private connectionTimeout: number | null = null;
  private cleanupInterval: number | null = null;
  private options: WebSocketServiceOptions;
  private binaryFragmentBuffer: BinaryFragmentBuffer = new BinaryFragmentBuffer();

  constructor(options: WebSocketServiceOptions = {}) {
    this.options = options;
  }

  connect(url: string) {
    // 避免重复连接
    if (this.ws && this.ws.readyState === WebSocket.OPEN) {
      addLog('WebSocket连接已存在且处于打开状态', 'info');
      return;
    }

    // 如果已有连接但处于其他状态，先关闭它
    if (this.ws) {
      try {
        addLog(`WebSocket当前状态: ${this.ws.readyState}`, 'info');
        this.ws.close();
      } catch (e) {
        addLog('关闭旧WebSocket连接时出错: ' + (e as Error).message, 'error');
      }
      this.ws = null;
    }

    this.url = url;
    addLog('正在连接WebSocket: ' + this.url, 'info');

    try {
      const ws = new WebSocket(this.url);
      // 设置二进制数据类型为ArrayBuffer而不是Blob
      ws.binaryType = 'arraybuffer';
      this.ws = ws;

      // 设置连接超时
      this.connectionTimeout = window.setTimeout(() => {
        if (ws.readyState === WebSocket.CONNECTING) {
          addLog('WebSocket连接超时（30秒未建立连接）。可能的原因：', 'error');
          addLog('1. 后端服务未启动或无法访问 (ws://localhost:8080/ws/dom-sync)', 'warn');
          addLog('2. 防火墙或网络配置阻止WebSocket连接', 'warn');
          addLog('3. 反向代理或负载均衡器未正确配置WebSocket支持', 'warn');
          addLog('4. 握手拦截器验证失败（如JWT Token验证失败）', 'warn');
          try {
            ws.close();
          } catch (closeError) {
            addLog('关闭WebSocket连接时出错: ' + (closeError as Error).message, 'error');
          }
          // 尝试重连
          this.attemptReconnect();
        }
      }, 30000); // 增加到30秒超时，给连接更多时间

      // 连接打开事件
      ws.onopen = () => {
        if (this.connectionTimeout) {
          clearTimeout(this.connectionTimeout);
          this.connectionTimeout = null;
        }
        this.reconnectAttempts = 0;
        addLog('WebSocket连接已建立，使用二进制协议', 'info');
        
        // 启动定期清理过期消息缓存
        if (this.cleanupInterval) {
          clearInterval(this.cleanupInterval);
        }
        this.cleanupInterval = window.setInterval(() => {
          this.binaryFragmentBuffer.cleanupExpired();
        }, 30000); // 每30秒检查一次
        
        if (this.options.onOpen) {
          this.options.onOpen();
        }
      };

      // 接收消息事件
      ws.onmessage = (event) => {
        try {
          addLog(`📥 WebSocket onmessage事件触发，数据类型: ${event.data.constructor.name}`, 'debug');
          
          // 只处理二进制消息
          if (event.data instanceof ArrayBuffer) {
            // 二进制消息处理
            addLog('接收到二进制消息，大小: ' + (event.data as ArrayBuffer).byteLength + ' 字节', 'debug');
            
            const success = handleBinaryMessage(
              event.data as ArrayBuffer,
              this.binaryFragmentBuffer,
              (decodedData: any, encoding: number) => {
                addLog(`📤 二进制消息处理完成，调用onMessage回调，数据类型: ${typeof decodedData}`, 'debug');
                // 消息完整接收并处理成功
                if (this.options.onMessage) {
                  this.options.onMessage(decodedData);
                }
              }
            );
            
            if (!success) {
              addLog('二进制消息处理失败', 'warn');
            }
          } else {
            addLog(`⚠️ 接收到非二进制消息，类型: ${typeof event.data}`, 'warn');
          }
        } catch (e) {
          addLog('处理WebSocket消息时发生异常：' + (e as Error).message, 'error');
          addLog('错误堆栈：' + (e as Error).stack, 'debug');
        }
      };

      // 连接关闭事件
      ws.onclose = (event) => {
        if (this.connectionTimeout) {
          clearTimeout(this.connectionTimeout);
          this.connectionTimeout = null;
        }
        addLog(`WebSocket连接已关闭，代码: ${event.code}, 原因: ${event.reason}`, 'info');

        // 检查关闭原因
        if (event.code === 1006) {
          addLog('WebSocket连接异常关闭（代码1006），可能原因：', 'warn');
          addLog('1. 网络中断或连接超时', 'info');
          addLog('2. 服务器主动断开连接', 'info');
          addLog('3. 防火墙、代理或中间件中断连接', 'info');
          
          // 添加Token状态检查
          this.checkTokenStatus();
        } else if (event.code !== 1000) {
          addLog(`WebSocket异常关闭（代码${event.code}），需要检查服务器日志`, 'warn');
        }

        // 只有在非正常关闭的情况下才尝试重连
        if (event.code !== 1000) { // 1000表示正常关闭
          if (this.reconnectAttempts < this.maxReconnectAttempts) {
            this.reconnectAttempts++;
            addLog(`WebSocket连接已断开，正在尝试第${this.reconnectAttempts}次重连...`, 'error');
            // 指数退避重连策略
            this.scheduleReconnect();
          } else {
            addLog('WebSocket连接已断开，已达到最大重连次数，停止重连', 'error');
            // 重置重连次数，以便用户手动重新连接
            this.reconnectAttempts = 0;
          }
        } else {
          addLog('WebSocket连接正常关闭，无需重连', 'info');
        }

        if (this.options.onClose) {
          this.options.onClose(event);
        }
      };

      // 连接错误事件
      ws.onerror = (error) => {
        if (this.connectionTimeout) {
          clearTimeout(this.connectionTimeout);
          this.connectionTimeout = null;
        }
        
        // 检查当前WebSocket状态来判断错误类型
        const readyState = ws.readyState;
        let errorDescription = '';
        
        if (readyState === WebSocket.CLOSED) {
          errorDescription = '握手失败：连接已关闭。通常是服务器拒绝连接（可能是认证失败、Token过期或无效）';
          
          // 添加Token状态检查
          this.checkTokenStatus();
        } else if (readyState === WebSocket.CLOSING) {
          errorDescription = '连接正在关闭中';
        } else if (readyState === WebSocket.CONNECTING) {
          errorDescription = '连接超时或连接被中断';
        } else {
          errorDescription = '未知错误';
        }
        
        addLog('WebSocket错误 [状态码: ' + readyState + ']：' + errorDescription, 'error');
        addLog('错误详情：' + (error as any).message, 'error');
        addLog('URL: ' + this.url, 'error');

        // 检查网络连接状态
        if (!navigator.onLine) {
          addLog('网络诊断：网络连接不可用，请检查您的网络连接', 'error');
        } else {
          addLog('网络诊断：网络连接正常，问题可能在服务器端', 'warn');
        }

        // 如果连接失败，尝试重新连接
        this.attemptReconnect();

        if (this.options.onError) {
          this.options.onError(error);
        }
      };
    } catch (e) {
      addLog('创建WebSocket连接失败: ' + (e as Error).message, 'error');
      addLog('详细信息: ' + (e as Error).stack, 'debug');
      addLog('诊断帮助:', 'info');
      addLog('1. 检查URL格式是否正确: ' + this.url, 'info');
      addLog('2. WebSocket创建失败可能指示浏览器不支持WebSocket或安全需求不满足', 'warn');
      addLog('3. 检查HTTPS页面是否使用了WSS协议', 'warn');
      // 如果创建连接失败，也尝试重连
      this.attemptReconnect();
    }
  }

  send(message: string) {
    if (this.ws && this.ws.readyState === WebSocket.OPEN) {
      try {
        this.ws.send(message);
        addLog('已发送指令：' + message, 'info');
      } catch (e) {
        addLog('发送指令失败：' + (e as Error).message, 'error');
      }
    } else {
      addLog('WebSocket连接已断开，无法发送指令', 'error');
      // 尝试重新连接
      this.attemptReconnect();
    }
  }

  close() {
    if (this.ws) {
      try {
        this.ws.close();
        addLog('WebSocket连接已关闭', 'info');
      } catch (e) {
        addLog('关闭WebSocket连接时出错: ' + (e as Error).message, 'error');
      }
      this.ws = null;
    }
    
    // 清理定时器
    if (this.cleanupInterval) {
      clearInterval(this.cleanupInterval);
      this.cleanupInterval = null;
    }
  }

  // 检查Token状态
  private checkTokenStatus() {
    const token = localStorage.getItem('token');
    const tokenValidation = TokenUtils.validateToken(token);
    
    if (!token) {
      addLog('错误：未找到认证Token，请重新登录', 'error');
    } else if (tokenValidation.isExpired) {
      addLog('错误：Token已过期，请重新登录', 'error');
    } else {
      addLog(`Token有效，将在 ${tokenValidation.minutesLeft?.toFixed(1)} 分钟后过期`, 'info');
    }
  }

  // 尝试重连
  private attemptReconnect() {
    if (this.reconnectAttempts < this.maxReconnectAttempts) {
      this.reconnectAttempts++;
      const delay = Math.min(this.reconnectDelay * Math.pow(2, this.reconnectAttempts - 1), 30000);
      addLog(`WebSocket连接失败，${delay}ms后进行第${this.reconnectAttempts}次重连...`, 'warn');
      this.scheduleReconnect(delay);
    } else {
      addLog('WebSocket连接失败，已达到最大重连次数，停止重连', 'error');
      if (this.url.includes('ws://localhost:8080/ws/dom-sync')) {
        addLog('诊断建议：', 'info');
        addLog('1. 检查后端服务是否正常运行（默认: ws://localhost:8080/ws/dom-sync）', 'info');
        addLog('2. 检查JWT Token是否过期（Token有效期: 2小时）', 'info');
        addLog('3. 检查浏览器控制台是否有详细的错误日志', 'info');
        addLog('4. 检查后端是否输出认证拒绝的原因日志', 'info');
        addLog('5. 可尝试手动刷新页面重新连接', 'info');
      }
      // 重置重连次数，以便用户手动重新连接
      this.reconnectAttempts = 0;
    }
  }

  // 安排重连
  private scheduleReconnect(delay?: number) {
    const reconnectDelay = delay ?? Math.min(this.reconnectDelay * Math.pow(2, this.reconnectAttempts - 1), 30000);
    setTimeout(() => this.connect(this.url), reconnectDelay);
  }

  isConnected(): boolean {
    return this.ws !== null && this.ws.readyState === WebSocket.OPEN;
  }
}