package pangea.hiagent.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import pangea.hiagent.workpanel.SseEventManager;
import pangea.hiagent.utils.UserUtils;
import pangea.hiagent.dto.WorkPanelEvent;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * 时间轴事件控制器
 * 提供ReAct过程的实时事件推送功能
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/agent")
@RequiredArgsConstructor
public class TimelineEventController {
    
    private final SseEventManager sseEventManager;
    private final ObjectMapper objectMapper;
    
    /**
     * 获取当前认证用户ID
     */
    private String getCurrentUserId() {
        return UserUtils.getCurrentUserId();
    }
    
    /**
     * 订阅时间轴事件
     * 支持 SSE (Server-Sent Events) 格式的实时事件推送
     * 
     * @return SSE emitter
     */
    @GetMapping("/timeline-events")
    public SseEmitter subscribeTimelineEvents() {
        log.info("开始处理时间轴事件订阅请求");
        
        String userId = getCurrentUserId();
        // 创建 SSE emitter
        SseEmitter emitter = new SseEmitter(300000L); // 5分钟超时
        
        if (userId == null) {
            log.error("用户未认证");
            // 使用sendError方法发送错误信息，而不是直接completeWithError
            sseEventManager.sendError(emitter, "用户未认证");
            return emitter;
        }
        
        log.debug("用户认证成功，用户ID: {}", userId);
        
        // 注册 emitter 回调
        emitter.onCompletion(() -> {
            log.debug("SSE连接完成");
            // 从所有Agent的emitter列表中移除
            sseEventManager.removeEmitter(emitter);
        });
        
        emitter.onTimeout(() -> {
            log.warn("SSE连接超时");
            try {
                emitter.complete();
            } catch (Exception e) {
                log.error("关闭SSE连接失败", e);
            }
            // 从所有Agent的emitter列表中移除
            sseEventManager.removeEmitter(emitter);
        });
        
        emitter.onError(throwable -> {
            log.error("SSE连接错误", throwable);
            // 从所有Agent的emitter列表中移除
            sseEventManager.removeEmitter(emitter);
        });
        
        // 发送连接成功的消息
        try {
            emitter.send(SseEmitter.event()
                    .name("connected")
                    .data("{\"type\":\"connected\",\"message\":\"连接成功\"}")
                    .build());
        } catch (IOException e) {
            log.error("发送连接成功消息失败", e);
        }
        
        log.info("时间轴事件订阅连接已建立，用户ID: {}", userId);
        return emitter;
    }
    
    /**
     * 广播事件给所有订阅者
     * 
     * @param event 事件数据
     */
    public void broadcastEvent(WorkPanelEvent event) {
        if (event == null) {
            log.warn("广播事件时接收到null事件");
            return;
        }
        
        try {
            // 构造事件数据
            String eventData = constructEventData(event);
            
            // 对工具事件添加详细的日志
            if ("tool_call".equals(event.getEventType()) || "tool_result".equals(event.getEventType()) || "tool_error".equals(event.getEventType())) {
                log.info("[工具事件] 类型={}, 工具={}, 有toolInput={}, 有toolOutput={}", 
                        event.getEventType(), 
                        event.getToolName(),
                        event.getToolInput() != null,
                        event.getToolOutput() != null);
                log.debug("[工具事件详情] toolInput={}, toolOutput={}", 
                        event.getToolInput(),
                        event.getToolOutput());
            }
            
            // 获取所有emitter并广播
            List<SseEmitter> emitters = sseEventManager.getEmitters();
            int successCount = 0;
            int failureCount = 0;
            
            for (SseEmitter emitter : new CopyOnWriteArrayList<>(emitters)) {
                try {
                    emitter.send(SseEmitter.event()
                            .name("message")
                            .data(eventData)
                            .build());
                    successCount++;
                } catch (IOException e) {
                    log.error("发送事件失败，移除失效连接: {}", e.getMessage());
                    emitters.remove(emitter);
                    failureCount++;
                }
            }
            
            if (failureCount > 0) {
                log.warn("事件广播部分失败: 成功={}, 失败={}", successCount, failureCount);
            }
        } catch (Exception e) {
            log.error("广播事件失败: 事件类型={}, 工具={}, 错误信息={}", 
                    event.getEventType(), 
                    event.getToolName(), 
                    e.getMessage(), 
                    e);
        }
    }
    
    /**
     * 构造事件数据JSON字符串
     * 
     * @param event 事件对象
     * @return JSON字符串
     */
    private String constructEventData(WorkPanelEvent event) {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        sb.append("\"type\":\"").append(sanitizeJsonString(event.getEventType())).append("\",");
        sb.append("\"timestamp\":").append(event.getTimestamp()).append(",");
        sb.append("\"title\":\"").append(sanitizeJsonString(getEventTitle(event))).append("\",");
        
        if (event.getContent() != null) {
            sb.append("\"content\":\"").append(sanitizeJsonString(event.getContent())).append("\",");
        }
        
        // 根据事件类型添加特定字段
        switch (event.getEventType()) {
            case "thinking":
                sb.append("\"thinkingType\":\"").append(sanitizeJsonString(event.getThinkingType())).append("\",");
                break;
            case "tool_call":
            case "tool_result":
            case "tool_error":
                sb.append("\"toolName\":\"").append(sanitizeJsonString(event.getToolName())).append("\",");
                sb.append("\"toolAction\":\"").append(sanitizeJsonString(event.getToolAction())).append("\",");
                
                // 正确序列化 toolInput 为 JSON 对象
                // 无论toolInput是否为null，都要包含这个字段以确保前端能正确识别
                try {
                    if (event.getToolInput() != null) {
                        String toolInputJson = objectMapper.writeValueAsString(event.getToolInput());
                        sb.append("\"toolInput\":").append(toolInputJson).append(",");
                        log.debug("[toolInput序列化成功] 工具={}, JSON={}", event.getToolName(), toolInputJson);
                    } else {
                        sb.append("\"toolInput\":null,");
                        log.debug("[toolInput为null] 工具={}, 类型={}", event.getToolName(), event.getEventType());
                    }
                } catch (Exception e) {
                    // 如果序列化失败，记录警告并回退到字符串表示
                    log.warn("[序列化toolInput失败] 工具={}, 错误={}, 已回退为字符串表示", event.getToolName(), e.getMessage());
                    if (event.getToolInput() != null) {
                        sb.append("\"toolInput\":\"").append(sanitizeJsonString(event.getToolInput().toString())).append("\",");
                    } else {
                        sb.append("\"toolInput\":null,");
                    }
                }
                
                // 正确序列化 toolOutput 为 JSON 对象
                // 无论toolOutput是否为null，都要包含这个字段以确保前端能正确识别
                try {
                    if (event.getToolOutput() != null) {
                        String toolOutputJson = objectMapper.writeValueAsString(event.getToolOutput());
                        sb.append("\"toolOutput\":").append(toolOutputJson).append(",");
                        log.debug("[toolOutput序列化成功] 工具={}, JSON={}", event.getToolName(), toolOutputJson);
                    } else {
                        sb.append("\"toolOutput\":null,");
                        log.debug("[toolOutput为null] 工具={}, 类型={}", event.getToolName(), event.getEventType());
                    }
                } catch (Exception e) {
                    // 如果序列化失败，记录警告并回退到字符串表示
                    log.warn("[序列化toolOutput失败] 工具={}, 错误={}, 已回退为字符串表示", event.getToolName(), e.getMessage());
                    if (event.getToolOutput() != null) {
                        sb.append("\"toolOutput\":\"").append(sanitizeJsonString(String.valueOf(event.getToolOutput()))).append("\",");
                    } else {
                        sb.append("\"toolOutput\":null,");
                    }
                }                
                if (event.getToolStatus() != null) {
                    sb.append("\"toolStatus\":\"").append(sanitizeJsonString(event.getToolStatus())).append("\",");
                }
                
                // 添加执行时间字段
                if (event.getExecutionTime() != null) {
                    sb.append("\"executionTime\":").append(event.getExecutionTime()).append(",");
                }
                break;
            case "log":
                sb.append("\"logLevel\":\"").append(sanitizeJsonString(event.getLogLevel())).append("\",");
                break;
            case "embed":
                if (event.getEmbedUrl() != null) {
                    sb.append("\"embedUrl\":\"").append(sanitizeJsonString(event.getEmbedUrl())).append("\",");
                }
                if (event.getEmbedType() != null) {
                    sb.append("\"embedType\":\"").append(sanitizeJsonString(event.getEmbedType())).append("\",");
                }
                if (event.getEmbedTitle() != null) {
                    sb.append("\"embedTitle\":\"").append(sanitizeJsonString(event.getEmbedTitle())).append("\",");
                }
                if (event.getEmbedHtmlContent() != null) {
                    sb.append("\"embedHtmlContent\":\"").append(sanitizeJsonString(event.getEmbedHtmlContent())).append("\",");
                }
                break;
        }
        
        // 移除最后一个逗号并添加结束括号
        if (sb.charAt(sb.length() - 1) == ',') {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        
        return sb.toString();
    }
    
    /**
     * 获取事件标题
     * 
     * @param event 事件对象
     * @return 标题字符串
     */
    private String getEventTitle(WorkPanelEvent event) {
        switch (event.getEventType()) {
            case "thinking":
                return "🧠 思考过程";
            case "tool_call":
                return "🎬 工具调用: " + (event.getToolName() != null ? event.getToolName() : "未知工具");
            case "tool_result":
                return "✅ 工具结果: " + (event.getToolName() != null ? event.getToolName() : "未知工具");
            case "tool_error":
                return "❌ 工具错误: " + (event.getToolName() != null ? event.getToolName() : "未知工具");
            case "log":
                return "📝 日志信息";
            default:
                return "📋 事件";
        }
    }
    
    /**
     * 清理JSON字符串中的特殊字符
     * 
     * @param str 原始字符串
     * @return 清理后的字符串
     */
    private String sanitizeJsonString(String str) {
        if (str == null) {
            return "";
        }
        return str.replace("\\", "\\\\")
                 .replace("\"", "\\\"")
                 .replace("\n", "\\n")
                 .replace("\r", "\\r")
                 .replace("\t", "\\t");
    }
}