package pangea.hiagent.agent.react;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.client.ChatClient;
import org.springframework.ai.chat.messages.*;
import org.springframework.ai.chat.model.ChatResponse;
import org.springframework.ai.chat.prompt.Prompt;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import pangea.hiagent.agent.service.ErrorHandlerService;
import pangea.hiagent.agent.service.TokenConsumerWithCompletion;
import pangea.hiagent.memory.MemoryService;
import pangea.hiagent.model.Agent;
import pangea.hiagent.tool.AgentToolManager;
import pangea.hiagent.tool.impl.DateTimeTools;
import java.util.List;
import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

/**
 * 简化的默认ReAct执行器实现
 */
@Slf4j
@Service
public class DefaultReactExecutor implements ReactExecutor {
    
    private static final String DEFAULT_SYSTEM_PROMPT = 
        "You are a powerful AI assistant powered by the ReAct (Reasoning + Acting) framework. Your task is to solve complex user queries by:\n" +
        "1. Breaking down problems into clear thinking steps\n" +
        "2. Intelligently selecting and combining the right tools to solve each sub-problem\n" +
        "3. Processing tool results and iterating until you reach the final answer\n\n" +
        "=== TOOL SYNERGY STRATEGY ===\n" +
        "You have access to multiple specialized tools. You should automatically think about tool combinations that create value:\n" +
        "- Chain tools together when one tool's output feeds into another tool's input\n" +
        "- Use text processing tools to prepare data before saving with file tools\n" +
        "- Use calculation tools within larger workflows to perform computations\n" +
        "- Combine multiple tools to enrich data and provide better insights\n" +
        "- Use data extraction tools with analysis tools for comprehensive results\n\n" +
        "Examples of tool synergy:\n" +
        "1. Process a file → analyze content with text tools → save results\n" +
        "2. Get current date/time → combine with formatting tools → format for display\n" +
        "3. Extract web content → process text → calculate statistics → generate charts\n" +
        "4. Perform calculations → convert to specific format → store results\n\n" +
        "=== ReAct THINKING PROCESS ===\n" +
        "Follow this step-by-step process for every problem:\n\n" +
        "Step 1 - THOUGHT: Analyze the user's query\n" +
        "  - Break down the problem into sub-tasks\n" +
        "  - Identify which tools or tool combinations could solve each sub-task\n" +
        "  - Consider tool synergies: can output from one tool feed into another?\n" +
        "  - Plan the sequence of tool calls if multiple tools are needed\n\n" +
        "Step 2 - ACTION: Execute the planned tools\n" +
        "  - Call the most appropriate tool(s) based on your analysis\n" +
        "  - Tools are executed automatically by the Spring AI framework\n" +
        "  - Multiple tools can be invoked in sequence if needed\n" +
        "  - Wait for tool execution results before proceeding\n\n" +
        "Step 3 - OBSERVATION: Analyze tool results\n" +
        "  - Examine the data returned from tools\n" +
        "  - Verify accuracy and completeness\n" +
        "  - Consider if additional tools are needed to enrich or validate the data\n" +
        "  - Identify patterns or insights from the results\n\n" +
        "Step 4 - FINAL ANSWER: Synthesize and respond\n" +
        "  - Combine all tool results into a coherent answer\n" +
        "  - Present information clearly and logically\n" +
        "  - If a tool combination provided better insights, explain how the tools worked together\n" +
        "  - Provide actionable conclusions based on the gathered information\n\n" +
        "=== RESPONSE FORMAT ===\n" +
        "When responding, follow this structure:\n\n" +
        "1. Thought: Explain your problem analysis and tool selection strategy\n" +
        "   - What sub-problems did you identify?\n" +
        "   - Which tools address each sub-problem?\n" +
        "   - What is the tool execution sequence?\n\n" +
        "2. Action: Describe which tools you're calling and why\n" +
        "   - Tool_Call: 1.[tool name] → Purpose: [why you're using it]\n" +
        "   - Tool_Call: 2.[tool name] → Purpose: [how it complements Tool 1]\n" +
        "   - Tool_Call: N.[tool name] → Purpose: [final enrichment or validation]\n\n" +
        "3. Observation: Interpret the results\n" +
        "   - What did each tool reveal?\n" +
        "   - How do the tool results relate to each other?\n" +
        "   - What patterns or insights emerged from combining tools?\n\n" +
        "4. Final_Answer: Provide the solution\n" +
        "   - Clear, natural language answer to the user's question\n" +
        "   - Highlight insights gained from tool combinations\n" +
        "   - Offer follow-up insights or recommendations\n\n" +
        "=== CRITICAL RULES ===\n" +
        "- ALWAYS think about tool combinations first - complex problems usually need multiple tools\n" +
        "- DO NOT limit yourself to single tools - maximize tool synergy\n" +
        "- ALWAYS wait for actual tool execution results, never make up data\n" +
        "- Tools are called automatically by Spring AI - focus on selecting the right tools\n" +
        "- Be explicit about your tool selection strategy in the Thought section\n" +
        "- If a tool is unavailable, explain what additional capabilities you would need\n" +
        "- Maintain conversational tone in Final Answer, not technical tool details";
    
    private final List<ReactCallback> reactCallbacks = new ArrayList<>();
    private final AtomicInteger stepCounter = new AtomicInteger(0);
    
    @Autowired
    private DateTimeTools dateTimeTools;
    
    @Autowired
    private MemoryService memoryService;
    
    @Autowired
    private ErrorHandlerService errorHandlerService;

    @Autowired
    private TokenTextSegmenter tokenTextSegmenter;
    
    private final AgentToolManager agentToolManager;
    
    public DefaultReactExecutor(AgentToolManager agentToolManager) {
        this.agentToolManager = agentToolManager;
    }
    
    @Override
    public void addReactCallback(ReactCallback callback) {
        if (callback != null) {
            reactCallbacks.add(callback);
        }
    }
    
    @Override
    public String execute(ChatClient chatClient, String userInput, List<Object> tools, Agent agent) {
        log.info("开始执行ReAct流程，用户输入: {}", userInput);
        
        stepCounter.set(0);
        
        List<Object> agentTools = getAgentTools(agent);
        
        try {
            triggerThinkStep("开始处理用户请求: " + userInput);
            
            Prompt prompt = buildPromptWithHistory(DEFAULT_SYSTEM_PROMPT, userInput, agent);
            
            ChatResponse response = chatClient.prompt(prompt)
                    .tools(agentTools.toArray())
                    .call()
                    .chatResponse();
            
            String responseText = response.getResult().getOutput().getText();
            
            triggerObservationStep(responseText);
            
            log.info("最终答案: {}", responseText);
            
            triggerFinalAnswerStep(responseText);
            
            return responseText;
        } catch (Exception e) {
            log.error("执行ReAct流程时发生错误", e);
            return handleReActError(e);
        }
    }
    
    /**
     * 处理ReAct执行过程中发生的错误
     * 
     * @param e 发生的异常
     * @return 错误处理结果
     */
    private String handleReActError(Exception e) {
        return errorHandlerService.handleSyncError(e, "处理ReAct请求时发生错误");
    }
    
    /**
     * 构建带有历史记录的提示词
     * 
     * @param systemPrompt 系统提示词
     * @param userInput 用户输入
     * @param agent 智能体对象
     * @return 构建好的提示词对象
     */
    private Prompt buildPromptWithHistory(String systemPrompt, String userInput, Agent agent) {
        List<org.springframework.ai.chat.messages.Message> messages = new ArrayList<>();
        
        messages.add(new SystemMessage(systemPrompt));
        
        if (agent != null) {
            try {
                String sessionId = memoryService.generateSessionId(agent);
                
                int historyLength = agent.getHistoryLength() != null ? agent.getHistoryLength() : 10;
                
                List<org.springframework.ai.chat.messages.Message> historyMessages = 
                    memoryService.getHistoryMessages(sessionId, historyLength);
                
                messages.addAll(historyMessages);
                
                memoryService.addUserMessageToMemory(sessionId, userInput);
            } catch (Exception e) {
                log.warn("获取历史对话记录时发生错误: {}", e.getMessage());
            }
        }
        
        messages.add(new UserMessage(userInput));
        
        return new Prompt(messages);
    }
    
    @Override
    public void executeStream(ChatClient chatClient, String userInput, List<Object> tools, Consumer<String> tokenConsumer, Agent agent) {
        log.info("使用stream()方法处理ReAct流程，支持真正的流式输出");
        
        stepCounter.set(0);
        
        List<Object> agentTools = getAgentTools(agent);
        
        StringBuilder fullResponse = new StringBuilder();
        
        try {
            triggerThinkStep("开始处理用户请求: " + userInput);
            
            Prompt prompt = buildPromptWithHistory(DEFAULT_SYSTEM_PROMPT, userInput, agent);
            
            chatClient.prompt(prompt)
                    .tools(agentTools.toArray())
                    .stream()
                    .chatResponse()
                    .subscribe(
                        chatResponse -> handleTokenResponse(chatResponse, tokenConsumer, fullResponse),
                        throwable -> handleStreamError(throwable, tokenConsumer),
                        () -> handleStreamCompletion(tokenConsumer, fullResponse, agent)
                    );
            
        } catch (Exception e) { 
            log.error("流式执行ReAct流程时发生错误", e);
            errorHandlerService.handleReactFlowError(e, tokenConsumer);
        }
    }
    
    /**
     * 处理流式响应中的单个token
     * 
     * @param chatResponse 聊天响应对象
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应构建器
     */
    private void handleTokenResponse(org.springframework.ai.chat.model.ChatResponse chatResponse, Consumer<String> tokenConsumer, StringBuilder fullResponse) {
        try {
            String token = chatResponse.getResult().getOutput().getText();
            
            if (isValidToken(token)) {
                fullResponse.append(token);
                
                // analyzeAndRecordToolEvents(token, fullResponse.toString());
                
                if (tokenConsumer != null) {
                    tokenConsumer.accept(token);
                }

                tokenTextSegmenter.inputChar(token);
                
                // 改进：在流式处理过程中实时解析关键词
                // processTokenForStepsWithFullResponse(token, fullResponse.toString());
            }
        } catch (Exception e) {
            log.error("处理token时发生错误", e);
        }
    }
    
    /**
     * 处理流式响应完成事件
     * 
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应内容
     * @param agent 智能体对象
     */
    private void handleStreamCompletion(Consumer<String> tokenConsumer, StringBuilder fullResponse, Agent agent) {
        try {
            log.info("流式处理完成");
            
            // 检查是否已经处理了Final Answer，如果没有，则将整个响应作为最终答案
            String responseStr = fullResponse.toString();
            if (!hasFinalAnswerBeenTriggered(responseStr)) {
                triggerFinalAnswerStep(responseStr);
            }
            
            saveAssistantResponseToMemory(agent, responseStr);
            
            sendCompletionEvent(tokenConsumer, responseStr);
        } catch (Exception e) {
            log.error("处理流式完成回调时发生错误", e);
            handleCompletionError(tokenConsumer, e);
        }
    }
    
    /**
     * 检查是否已经触发了Final Answer步骤
     * 
     * @param fullResponse 完整响应内容
     * @return 如果已经触发了Final Answer则返回true，否则返回false
     */
    private boolean hasFinalAnswerBeenTriggered(String fullResponse) {
        String[] finalAnswerPatterns = {"Final Answer:", "final answer:", "FINAL ANSWER:", "Final_Answer:", "final_answer:", "FINAL_ANSWER:", "最终答案:"};
        for (String pattern : finalAnswerPatterns) {
            if (fullResponse.toLowerCase().contains(pattern.toLowerCase())) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * 将助手的回复保存到内存中
     * 
     * @param agent 智能体对象
     * @param response 助手的回复内容
     */
    private void saveAssistantResponseToMemory(Agent agent, String response) {
        if (agent != null) {
            try {
                String sessionId = memoryService.generateSessionId(agent);
                memoryService.addAssistantMessageToMemory(sessionId, response);
            } catch (Exception e) {
                log.warn("保存助理回复到内存时发生错误: {}", e.getMessage());
            }
        }
    }
    
    /**
     * 处理完成事件时发生的错误
     * 
     * @param tokenConsumer token消费者
     * @param e 发生的异常
     */
    private void handleCompletionError(Consumer<String> tokenConsumer, Exception e) {
        if (tokenConsumer instanceof TokenConsumerWithCompletion) {
            try {
                String errorId = errorHandlerService.generateErrorId();
                String fullErrorMessage = errorHandlerService.buildFullErrorMessage("处理完成时发生错误", e, errorId, "ReAct");
                ((TokenConsumerWithCompletion) tokenConsumer).onComplete("[" + errorId + "] " + fullErrorMessage);
            } catch (Exception ex) {
                log.error("调用onComplete时发生错误", ex);
            }
        }
    }
    
    /**
     * 验证token是否有效
     * 
     * @param token 待验证的token
     * @return 如果token有效则返回true，否则返回false
     */
    private boolean isValidToken(String token) {
        return token != null && !token.isEmpty();
    }
    
    /**
     * 基于完整响应进行关键词解析
     * 
     * @param token 当前token
     * @param fullResponse 完整响应内容
     */
    private void processTokenForStepsWithFullResponse(String token, String fullResponse) {
        if (fullResponse == null || fullResponse.isEmpty()) {
            return;
        }
        
        // 检查并处理Thought部分
        processThoughtSection(fullResponse);
        
        // 检查并处理Action部分
        processActionSection(fullResponse);
        
        // 检查并处理Observation部分
        processObservationSection(fullResponse);
        
        // 检查并处理Final Answer部分
        processFinalAnswerSection(fullResponse);
    }
    
    /**
     * 处理Thought部分
     * 
     * @param fullResponse 完整响应内容
     */
    private void processThoughtSection(String fullResponse) {
        // 检查是否包含Thought标记
        String[] thoughtPatterns = {"Thought:", "thought:", "THOUGHT:"};
        for (String pattern : thoughtPatterns) {
            if (fullResponse.toLowerCase().contains(pattern.toLowerCase())) {
                // 检查是否已经有对应的步骤被触发，避免重复触发
                if (!isStepAlreadyTriggered(ReactStepType.THOUGHT, fullResponse, pattern)) {
                    String content = extractContentAfterKeyword(fullResponse, pattern);
                    // 如果内容中包含下一个步骤的关键词，则只提取到下一个关键词之前的内容
                    String[] nextKeywords = {"Action:", "Observation:", "Final Answer:"};
                    for (String nextKeyword : nextKeywords) {
                        int nextKeywordIndex = content.toLowerCase().indexOf(nextKeyword.toLowerCase());
                        if (nextKeywordIndex != -1) {
                            content = content.substring(0, nextKeywordIndex).trim();
                            break;
                        }
                    }
                    if (!content.isEmpty()) {
                        triggerThinkStep(content);
                    }
                }
                break; // 找到一个就跳出
            }
        }
    }
    
    /**
     * 处理Action部分
     * 
     * @param fullResponse 完整响应内容
     */
    private void processActionSection(String fullResponse) {
        // 检查是否包含Action标记
        String[] actionPatterns = {"Action:", "action:", "ACTION:"};
        for (String pattern : actionPatterns) {
            if (fullResponse.toLowerCase().contains(pattern.toLowerCase())) {
                if (!isStepAlreadyTriggered(ReactStepType.ACTION, fullResponse, pattern)) {
                    String content = extractContentAfterKeyword(fullResponse, pattern);
                    String[] nextKeywords = {"Observation:", "Thought:", "Final Answer:"};
                    for (String nextKeyword : nextKeywords) {
                        int nextKeywordIndex = content.toLowerCase().indexOf(nextKeyword.toLowerCase());
                        if (nextKeywordIndex != -1) {
                            content = content.substring(0, nextKeywordIndex).trim();
                            break;
                        }
                    }
                    // 尝试提取工具名称和参数
                    String toolName = extractToolName(fullResponse); // 从整个响应中提取，因为可能在Action后有详细信息
                    Object toolArgs = extractToolArgs(fullResponse);
                    
                    if (toolName != null && !toolName.isEmpty()) {
                        triggerActionStep(toolName, "tool_call", toolArgs);
                    } else {
                        triggerActionStep("unknown", "tool_call", toolArgs);
                    }
                }
                break;
            }
        }
    }
    
    /**
     * 处理Observation部分
     * 
     * @param fullResponse 完整响应内容
     */
    private void processObservationSection(String fullResponse) {
        // 检查是否包含Observation标记
        String[] observationPatterns = {"Observation:", "observation:", "OBSERVATION:"};
        for (String pattern : observationPatterns) {
            if (fullResponse.toLowerCase().contains(pattern.toLowerCase())) {
                if (!isStepAlreadyTriggered(ReactStepType.OBSERVATION, fullResponse, pattern)) {
                    String content = extractContentAfterKeyword(fullResponse, pattern);
                    String[] nextKeywords = {"Thought:", "Action:", "Final Answer:"};
                    for (String nextKeyword : nextKeywords) {
                        int nextKeywordIndex = content.toLowerCase().indexOf(nextKeyword.toLowerCase());
                        if (nextKeywordIndex != -1) {
                            content = content.substring(0, nextKeywordIndex).trim();
                            break;
                        }
                    }
                    if (!content.isEmpty()) {
                        triggerObservationStep(content);
                    }
                }
                break;
            }
        }
    }
    
    /**
     * 处理Final Answer部分
     * 
     * @param fullResponse 完整响应内容
     */
    private void processFinalAnswerSection(String fullResponse) {
        // 检查是否包含Final Answer标记（包括带下划线和不带下划线的变体）
        String[] finalAnswerPatterns = {"Final Answer:", "final answer:", "FINAL ANSWER:", "Final_Answer:", "final_answer:", "FINAL_ANSWER:", "最终答案:"};
        for (String pattern : finalAnswerPatterns) {
            if (fullResponse.toLowerCase().contains(pattern.toLowerCase())) {
                if (!isStepAlreadyTriggered(ReactStepType.FINAL_ANSWER, fullResponse, pattern)) {
                    String content = extractContentAfterKeyword(fullResponse, pattern);
                    // Final Answer后的内容即为最终答案，不需要再查找下一个关键词
                    if (!content.isEmpty()) {
                        triggerFinalAnswerStep(content);
                    }
                }
                break;
            }
        }
    }
    
    /**
     * 检查某个步骤是否已经触发过，避免重复触发
     * 
     * @param stepType 步骤类型
     * @param fullResponse 完整响应内容
     * @param keyword 关键词
     * @return 如果步骤已经触发过则返回true，否则返回false
     */
    private boolean isStepAlreadyTriggered(ReactStepType stepType, String fullResponse, String keyword) {
        // 这里可以根据已有的步骤计数或内容来判断是否已经触发过
        // 为了简化，我们使用一个简单的检查：如果关键词在响应中出现多次，但只处理第一次
        int firstIndex = fullResponse.toLowerCase().indexOf(keyword.toLowerCase());
        int nextIndex = fullResponse.toLowerCase().indexOf(keyword.toLowerCase(), firstIndex + 1);
        
        // 如果没有找到关键词，返回true（无需触发）
        if (firstIndex == -1) {
            return true; // 没有关键词，无需触发
        }
        
        // 简单的检查：如果当前响应中包含关键词，且我们正在处理它，则认为未重复
        // 在这个简化版本中，我们假设每次检测到关键词时都处理它
        return false;
    }
    
    /**
     * 处理流式响应中的错误
     * 
     * @param throwable 异常对象
     * @param tokenConsumer token消费者
     */
    private void handleStreamError(Throwable throwable, Consumer<String> tokenConsumer) {
        errorHandlerService.handleStreamError(throwable, tokenConsumer, "ReAct流式处理");
    }
    
    /**
     * 发送完成事件
     * 
     * @param tokenConsumer token消费者
     * @param fullResponse 完整响应内容
     */
    private void sendCompletionEvent(Consumer<String> tokenConsumer, String fullResponse) {
        if (fullResponse == null) {
            fullResponse = "";
        }
        
        if (tokenConsumer instanceof TokenConsumerWithCompletion) {
            ((TokenConsumerWithCompletion) tokenConsumer).onComplete(fullResponse);
        } else if (tokenConsumer != null) {
            tokenConsumer.accept("");
        }
    }
    
    /**
     * 分析并记录工具事件
     * 
     * @param token 当前token
     * @param fullResponse 完整响应内容
     */
    // private void analyzeAndRecordToolEvents(String token, String fullResponse) {
    //     if (token == null || token.isEmpty()) {
    //         return;
    //     }
        
    //     String lowerToken = token.toLowerCase();
        
    //     if (lowerToken.startsWith("action:") || (lowerToken.contains("action:") && !fullResponse.contains("Observation:"))) {
    //         String toolName = extractToolName(token);
    //         Object toolArgs = extractToolArgs(token);
            
    //         if (toolName != null && !toolName.isEmpty()) {
    //             log.info("[ReAct Action] 检测到工具调用: {} 参数: {}", toolName, toolArgs);
    //         }
    //     }
        
    //     if (lowerToken.startsWith("observation:") || (lowerToken.contains("observation:") && fullResponse.contains("Observation:"))) {
    //         log.info("[ReAct Observation] 检测到工具结果: {}", token.substring(0, Math.min(100, token.length())));
    //     }
        
    //     if (lowerToken.startsWith("final answer:") || lowerToken.contains("final answer:")) {
    //         log.info("[ReAct Final Answer] 检测到最终答案");
    //     }
    // }
    
    /**
     * 从文本中提取工具名称
     * 
     * @param text 待解析的文本
     * @return 工具名称，如果未找到则返回null
     */
    private String extractToolName(String text) {
        if (text == null) return null;
        
        String[] patterns = {
            "Action:\\s*(\\w+)(?:\\(.*?\\))?$", // 匹配Action: toolName格式
            "Action:\\s*(\\w+)$", // 匹配Action: toolName格式
            "Action Input:\\s*\\{\\s*\"name\"\\s*:\\s*\"(\\w+)\"", // 匹配JSON中的工具名
            "Tool:\\s*(\\w+)",
            "tool:\\s*(\\w+)",
            "Calling\\s+(\\w+)",
            "calling\\s+(\\w+)",
            "Use\\s+(\\w+)",
            "use\\s+(\\w+)"
        };
        
        for (String pattern : patterns) {
            java.util.regex.Pattern p = java.util.regex.Pattern.compile(pattern, java.util.regex.Pattern.MULTILINE | java.util.regex.Pattern.CASE_INSENSITIVE);
            java.util.regex.Matcher m = p.matcher(text);
            if (m.find()) {
                return m.group(1);
            }
        }
        
        return null;
    }
    
    /**
     * 从文本中提取工具参数
     * 
     * @param text 待解析的文本
     * @return 工具参数，如果未找到则返回null
     */
    private Object extractToolArgs(String text) {
        if (text == null) return null;
            
        try {
            java.util.regex.Pattern actionInputPattern = java.util.regex.Pattern.compile("Action Input:\s*(.*)");
            java.util.regex.Matcher actionInputMatcher = actionInputPattern.matcher(text);
                
            if (actionInputMatcher.find()) {
                String inputStr = actionInputMatcher.group(1).trim();
                if (!inputStr.isEmpty() && inputStr.startsWith("{")) {
                    return parseJsonToMap(inputStr);
                }
            }
                
            java.util.regex.Pattern jsonPattern = java.util.regex.Pattern.compile("\\\"arguments\\\"\\s*:\\s*(\\{[^}]*\\})");
            java.util.regex.Matcher jsonMatcher = jsonPattern.matcher(text);
            if (jsonMatcher.find()) {
                return parseJsonToMap(jsonMatcher.group(1));
            }
                
        } catch (Exception e) {
            log.debug("解析工具参数时出错: {}", e.getMessage());
        }
            
        java.util.Map<String, Object> args = new java.util.HashMap<>();
        return args.isEmpty() ? null : args;
    }
    
    /**
     * 将JSON字符串解析为Map
     * 
     * @param jsonString JSON字符串
     * @return 解析后的Map对象
     */
    private java.util.Map<String, Object> parseJsonToMap(String jsonString) {
        try {
            jsonString = jsonString.trim().replaceAll("^\\{|\\}$", "").trim();
            java.util.Map<String, Object> map = new java.util.HashMap<>();
                
            if (!jsonString.isEmpty()) {
                String[] pairs = jsonString.split(",");
                for (String pair : pairs) {
                    String[] keyValue = pair.split(":", 2);
                    if (keyValue.length == 2) {
                        String key = keyValue[0].trim().replaceAll("^\"|\"$", "");
                        String value = keyValue[1].trim().replaceAll("^\"|\"$", "");
                        map.put(key, value);
                    }
                }
            }
            return map;
        } catch (Exception e) {
            log.debug("JSON解析失败: {}", e.getMessage());
            java.util.Map<String, Object> args = new java.util.HashMap<>();
            args.put("raw", jsonString);
            return args;
        }
    }
    
    /**
     * 提取关键词后的内容
     * 
     * @param text 原始文本
     * @param keyword 关键词
     * @return 关键词后的内容
     */
    private String extractContentAfterKeyword(String text, String keyword) {
        if (text == null || keyword == null) {
            return "";
        }
        
        // 找到关键词的位置
        int keywordIndex = -1;
        String lowerText = text.toLowerCase();
        String lowerKeyword = keyword.toLowerCase();
        
        // 首先尝试精确匹配
        keywordIndex = lowerText.indexOf(lowerKeyword);
        if (keywordIndex == -1) {
            // 如果没找到，尝试匹配带空格的版本
            keywordIndex = lowerText.indexOf(lowerKeyword.trim());
        }
        
        if (keywordIndex != -1) {
            // 提取关键词后的部分
            String content = text.substring(keywordIndex + keyword.length()).trim();
            
            // 如果内容以冒号开头，去掉它
            if (content.startsWith(":")) {
                content = content.substring(1).trim();
            }
            
            return content;
        }
        
        return text.trim();
    }
    
    /**
     * 触发思考步骤
     * 
     * @param content 思考内容
     */
    private void triggerThinkStep(String content) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.THOUGHT, content);
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发行动步骤
     * 
     * @param toolName 工具名称
     * @param toolAction 工具操作
     * @param toolArgs 工具参数
     */
    private void triggerActionStep(String toolName, String toolAction, Object toolArgs) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.ACTION, "执行工具: " + toolName);
        
        ReactStep.ToolCallAction toolActionObj = new ReactStep.ToolCallAction(toolName, toolArgs);
        reactStep.setAction(toolActionObj);
        
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发观察步骤
     * 
     * @param observation 观察内容
     */
    private void triggerObservationStep(String observation) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.OBSERVATION, observation);
        
        ReactStep.ToolObservation toolObservation = new ReactStep.ToolObservation(observation);
        reactStep.setObservation(toolObservation);
        
        notifyCallbacks(reactStep);
    }
    
    /**
     * 触发最终答案步骤
     * 
     * @param finalAnswer 最终答案
     */
    private void triggerFinalAnswerStep(String finalAnswer) {
        int stepNumber = stepCounter.incrementAndGet();
        ReactStep reactStep = new ReactStep(stepNumber, ReactStepType.FINAL_ANSWER, finalAnswer);
        notifyCallbacks(reactStep);
    }
    
    /**
     * 通知所有回调
     * 
     * @param reactStep ReAct步骤
     */
    private void notifyCallbacks(ReactStep reactStep) {
        for (ReactCallback callback : reactCallbacks) {
            try {
                callback.onStep(reactStep);
            } catch (Exception e) {
                log.error("执行ReAct回调时发生错误", e);
            }
        }
    }
    
    /**
     * 获取智能体工具
     * 
     * @param agent 智能体对象
     * @return 智能体可用的工具列表
     */
    private List<Object> getAgentTools(Agent agent) {
        if (agent == null) {
            List<Object> defaultTools = new ArrayList<>();
            defaultTools.add(dateTimeTools);
            return defaultTools;
        }
        
        try {
            List<Object> tools = agentToolManager.getAvailableToolInstances(agent);
            
            if (dateTimeTools != null && !tools.contains(dateTimeTools)) {
                tools.add(dateTimeTools);
            }
            
            return tools;
        } catch (Exception e) {
            log.error("获取工具实例时发生错误: {}", e.getMessage());
            List<Object> fallbackTools = new ArrayList<>();
            fallbackTools.add(dateTimeTools);
            return fallbackTools;
        }
    }
}