package pangea.hiagent.agent.processor;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.client.ChatClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import pangea.hiagent.agent.react.ReactCallback;
import pangea.hiagent.agent.react.ReactExecutor;
import pangea.hiagent.model.Agent;
import pangea.hiagent.rag.RagService;
import pangea.hiagent.tool.AgentToolManager;
import pangea.hiagent.web.dto.AgentRequest;
import pangea.hiagent.web.service.AgentService;

import java.util.List;
import java.util.function.Consumer;

/**
 * ReAct Agent处理器实现类
 * 职责：处理ReAct模式的Agent请求，集成ReAct执行器，支持工具调用和复杂推理
 */
@Slf4j
@Service
public class ReActAgentProcessor extends BaseAgentProcessor {
    
    @Autowired
    private AgentService agentService;
    
    @Autowired
    private RagService ragService;
    
    @Autowired
    private ReactCallback defaultReactCallback;
    
    @Autowired
    private ReactExecutor defaultReactExecutor;
    
    @Autowired
    private AgentToolManager agentToolManager;
    
    @Override
    public String processRequest(Agent agent, AgentRequest request, String userId) {
        log.info("使用ReAct Agent处理请求");
        return processRequestInternal(agent, request.getUserMessage(), userId);
    }
    
    @Override
    public void processStreamRequest(AgentRequest request, Agent agent, String userId, Consumer<String> tokenConsumer) {
        log.info("使用ReAct Agent处理流式请求");
        processRequestStreamInternal(agent, request.getUserMessage(), tokenConsumer, userId);
    }
    
    @Override
    public String getProcessorType() {
        return "react";
    }
    
    /**
     * 处理用户请求的主方法（同步方式）
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @param userId 用户ID（可选）
     * @return 处理结果
     */
    private String processRequestInternal(Agent agent, String userMessage, String userId) {
        log.info("开始处理ReAct Agent请求，Agent ID: {}, 用户消息: {}", agent != null ? agent.getId() : "null", userMessage);
        
        try {
            // 处理请求的通用前置逻辑
            String ragResponse = handlePreProcessing(agent, userMessage, userId, ragService, null);
            if (ragResponse != null) {
                // 触发最终答案回调
                if (defaultReactCallback != null) {
                    defaultReactCallback.onFinalAnswer(ragResponse);
                }
                return ragResponse;
            }
            
            // 准备执行环境
            ChatClient client = ChatClient.builder(agentService.getChatModelForAgent(agent)).build();
            List<Object> tools = agentToolManager.getAvailableToolInstances(agent);
            
            // 添加自定义回调到ReAct执行器
            if (defaultReactExecutor != null && defaultReactCallback != null) {
                defaultReactExecutor.addReactCallback(defaultReactCallback);
            }
            
            // 使用ReAct执行器执行流程，传递Agent对象以支持记忆功能
            String finalAnswer = defaultReactExecutor.execute(client, userMessage, tools, agent);
            
            // 将助理回复添加到ChatMemory
            String sessionId = generateSessionId(agent, userId);
            addAssistantMessageToMemory(sessionId, finalAnswer);
            
            return finalAnswer;
        } catch (Exception e) {
            return agentErrorHandler.handleSyncError(e, "处理ReAct请求时发生错误");
        }
    }
    
    /**
     * 处理用户请求的主方法（流式方式）
     * 
     * @param agent Agent对象
     * @param userMessage 用户消息
     * @param tokenConsumer token消费者
     * @param userId 用户ID（可选）
     */
    private void processRequestStreamInternal(Agent agent, String userMessage, Consumer<String> tokenConsumer, String userId) {
        log.info("开始流式处理ReAct Agent请求，Agent ID: {}, 用户消息: {}", agent != null ? agent.getId() : "null", userMessage);
        
        try {
            // 处理请求的通用前置逻辑
            String ragResponse = handlePreProcessing(agent, userMessage, userId, ragService, tokenConsumer);
            if (ragResponse != null) {
                // 触发最终答案回调
                if (defaultReactCallback != null) {
                    defaultReactCallback.onFinalAnswer(ragResponse);
                }
                return;
            }
            
            // 准备执行环境
            ChatClient client = ChatClient.builder(agentService.getChatModelForAgent(agent)).build();
            List<Object> tools = agentToolManager.getAvailableToolInstances(agent);
            
            // 添加自定义回调到ReAct执行器
            if (defaultReactExecutor != null && defaultReactCallback != null) {
                defaultReactExecutor.addReactCallback(defaultReactCallback);
            }
            
            // 检查模型是否支持流式输出
            if (agentService.getChatModelForAgent(agent) == null) {
                log.error("无法获取Agent的聊天模型");
                handleModelNotAvailable(tokenConsumer);
                return;
            }
            
            // 使用ReAct执行器流式执行流程，传递Agent对象以支持记忆功能
            defaultReactExecutor.executeStream(client, userMessage, tools, tokenConsumer, agent);
        } catch (Exception e) {
            agentErrorHandler.handleStreamError(e, tokenConsumer, "流式处理ReAct请求时发生错误");
            agentErrorHandler.ensureCompletionCallback(tokenConsumer, "处理请求时发生错误: " + e.getMessage());
        }
    }
    
    /**
     * 处理模型不可用的情况
     * 
     * @param tokenConsumer token消费者
     */
    private void handleModelNotAvailable(Consumer<String> tokenConsumer) {
        String errorMessage = "[错误] 无法获取Agent的聊天模型";
        // 发送错误信息
        agentErrorHandler.sendErrorMessage(tokenConsumer, errorMessage);
        // 确保在异常情况下也调用完成回调
        agentErrorHandler.ensureCompletionCallback(tokenConsumer, errorMessage);
    }
}