package pangea.hiagent.agent.react;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.workpanel.IWorkPanelDataCollector;

@Slf4j
@Component
public class TokenTextSegmenter {

    @Autowired
    private IWorkPanelDataCollector workPanelCollector;

    // 定义分段标识（按出现优先级排序）
    private static final String[] SEGMENT_MARKERS = {
            "Thought:",
            "Action:",
            "Observation:",
            "Final_Answer:"
    };

    // 当前缓存的输入字符
    private StringBuilder currentInputBuffer;
    // 已匹配到的分段标识
    private String matchedMarker;
    // 分段内容起始索引
    private int segmentContentStartIndex;

    public TokenTextSegmenter() {
        currentInputBuffer = new StringBuilder();
        matchedMarker = null;
        segmentContentStartIndex = 0;
    }

    /**
     * 逐字输入字符并处理分段
     * 
     * @param inputChar 单个输入字符
     * @return 当分割出有效文本段时返回该段内容，否则返回null
     */
    public void inputChar(String inputChar) {
        // 将字符加入缓存
        currentInputBuffer.append(inputChar);
        String currentBufferStr = currentInputBuffer.toString();

        // 1. 未匹配到标识时，检测是否出现分段标识
        if (matchedMarker == null) {
            for (String marker : SEGMENT_MARKERS) {
                if (currentBufferStr.endsWith(marker)) {
                    // 匹配到标识，记录信息
                    matchedMarker = marker;
                    segmentContentStartIndex = currentBufferStr.length();
                    // 输出标识本身（可选，根据需求决定是否包含标识）
                    log.info("【识别到分段标识】: {}", matchedMarker);
                }
            }
        }
        // 2. 已匹配到标识，检测是否出现下一个标识（或文本结束）
        else {
            for (String marker : SEGMENT_MARKERS) {
                if (!marker.equals(matchedMarker) && currentBufferStr.contains(marker)) {
                    // 找到下一个标识，截取当前分段内容
                    int nextMarkerStartIndex = currentBufferStr.indexOf(marker);
                    String segmentContent = currentBufferStr.substring(segmentContentStartIndex, nextMarkerStartIndex)
                            .trim();
                    // 重置状态，准备处理下一个分段
                    resetSegmentState(nextMarkerStartIndex);
                    // 输出当前分段内容
                    outputSegment(matchedMarker, segmentContent);
                }
            }
        }
    }

    /**
     * 文本输入结束时，处理最后一个分段
     * 
     * @return 最后一个分段的内容，无则返回null
     */
    public void finishInput() {
        if (matchedMarker != null && segmentContentStartIndex < currentInputBuffer.length()) {
            String lastSegmentContent = currentInputBuffer.substring(segmentContentStartIndex).trim();
            resetSegmentState(currentInputBuffer.length());
            outputSegment(matchedMarker, lastSegmentContent);
        }
    }

    /**
     * 重置分段状态
     * 
     * @param newStartIndex 新的起始索引
     */
    private void resetSegmentState(int newStartIndex) {
        // 保留未处理的部分，用于下一个分段
        String remainingStr = currentInputBuffer.substring(newStartIndex);
        currentInputBuffer = new StringBuilder(remainingStr);
        matchedMarker = null;
        segmentContentStartIndex = 0;
    }

    /**
     * 输出分段内容（封装输出逻辑）
     * 
     * @param marker  分段标识
     * @param content 分段内容
     * @return 格式化后的分段内容
     */
    private void outputSegment(String marker, String content) {
        log.info("【分段内容】{}: {}", marker, content);
        workPanelCollector.addEvent(null);
    }
}
