// iframe服务
import { addLog } from '@/utils/logUtils';
import { debounce, throttle } from '@/utils/functionUtils';
import { getElementSelector } from '@/utils/domUtils';

export class IframeService {
  private domViewRef: HTMLIFrameElement | null = null;
  private eventListeners: {
    [key: string]: EventListenerOrEventListenerObject;
  } = {};
  private mutationObserver: MutationObserver | null = null;

  constructor(domViewRef: HTMLIFrameElement | null) {
    this.domViewRef = domViewRef;
  }

  // 为iframe内容添加事件监听器
  addEventListeners(
    onClick: (selector: string) => void,
    onDoubleClick: (selector: string) => void,
    onInput: (selector: string, value: string) => void,
    onSubmit: (selector: string) => void,
    onScroll: (scrollY: number) => void,
    onSelect: (selector: string, value: string) => void,
    onKeyDown: (key: string) => void,
    onKeyUp: (key: string) => void,
    onKeyPress: (key: string) => void,
    onHover: (selector: string) => void,
    onFocus: (selector: string) => void,
    onBlur: (selector: string) => void
  ) {
    if (!this.domViewRef || !this.domViewRef.contentDocument) {
      addLog('无法访问iframe内容文档', 'error');
      return;
    }

    const iframeDoc = this.domViewRef.contentDocument;
    
    try {
      // 先移除已存在的监听器，避免重复绑定
      this.removeEventListeners();
      
      // 定义事件处理器配置
      const eventHandlers = [
        {
          type: 'click',
          handler: (event: Event) => {
            const target = event.target as HTMLElement;
            if (target) {
              const selector = getElementSelector(target);
              onClick(selector);
            }
          },
          options: { passive: true }
        },
        {
          type: 'dblclick',
          handler: (event: Event) => {
            const target = event.target as HTMLElement;
            if (target) {
              const selector = getElementSelector(target);
              onDoubleClick(selector);
            }
          },
          options: { passive: true }
        },
        {
          type: 'input',
          handler: debounce((event: Event) => {
            const target = event.target as HTMLInputElement;
            if (target && (target.tagName === 'INPUT' || target.tagName === 'TEXTAREA')) {
              const selector = getElementSelector(target);
              onInput(selector, target.value);
            }
          }, 500),
          options: { passive: true }
        },
        {
          type: 'submit',
          handler: (event: Event) => {
            const target = event.target as HTMLFormElement;
            if (target) {
              event.preventDefault(); // 阻止默认提交行为
              const selector = getElementSelector(target);
              onSubmit(selector);
            }
          },
          options: {}
        },
        {
          type: 'scroll',
          handler: throttle((event: Event) => {
            const target = event.target as HTMLElement;
            if (target) {
              const scrollY = target.scrollTop || 0;
              onScroll(scrollY);
            }
          }, 200),
          options: { passive: true }
        },
        {
          type: 'keydown',
          handler: (event: KeyboardEvent) => {
            const target = event.target as HTMLElement;
            if (target) {
              onKeyDown(event.key);
            }
          },
          options: { passive: true }
        },
        {
          type: 'keyup',
          handler: (event: KeyboardEvent) => {
            const target = event.target as HTMLElement;
            if (target) {
              onKeyUp(event.key);
            }
          },
          options: { passive: true }
        },
        {
          type: 'keypress',
          handler: (event: KeyboardEvent) => {
            const target = event.target as HTMLElement;
            if (target) {
              onKeyPress(event.key);
            }
          },
          options: { passive: true }
        },
        {
          type: 'mouseover',
          handler: (event: Event) => {
            const target = event.target as HTMLElement;
            if (target) {
              const selector = getElementSelector(target);
              onHover(selector);
            }
          },
          options: { passive: true }
        },
        {
          type: 'change',
          handler: (event: Event) => {
            const target = event.target as HTMLSelectElement;
            if (target && target.tagName === 'SELECT') {
              const selector = getElementSelector(target);
              const selectedValue = target.value;
              onSelect(selector, selectedValue);
            }
          },
          options: { passive: true }
        },
        {
          type: 'focus',
          handler: (event: Event) => {
            const target = event.target as HTMLElement;
            if (target) {
              const selector = getElementSelector(target);
              onFocus(selector);
            }
          },
          options: true // 使用捕获阶段
        },
        {
          type: 'blur',
          handler: (event: Event) => {
            const target = event.target as HTMLElement;
            if (target) {
              const selector = getElementSelector(target);
              onBlur(selector);
            }
          },
          options: true // 使用捕获阶段
        }
      ];
      
      // 注册所有事件监听器
      eventHandlers.forEach(({ type, handler, options }) => {
        this.eventListeners[type] = handler;
        iframeDoc.addEventListener(type, handler, options);
      });
      
      // 添加MutationObserver监控DOM变化
      // 确保body元素存在后再创建观察器
      if (iframeDoc.body) {
        this.mutationObserver = new MutationObserver(() => {
          // 当DOM发生变化时，重新绑定事件监听器以确保新元素能响应事件
        });
        
        this.mutationObserver.observe(iframeDoc.body, {
          childList: true,
          subtree: true
        });
      } else {
        addLog('iframe body元素尚未准备就绪，跳过MutationObserver设置', 'warn');
      }
      
      addLog('已为iframe添加事件监听器', 'info');
    } catch (e) {
      addLog('为iframe添加事件监听器失败: ' + (e as Error).message, 'error');
    }
  }

  // 移除iframe事件监听器
  removeEventListeners() {
    if (!this.domViewRef || !this.domViewRef.contentDocument) {
      return;
    }

    const iframeDoc = this.domViewRef.contentDocument;
    
    try {
      // 定义需要移除的事件类型
      const eventTypes = [
        { type: 'click', useCapture: false },
        { type: 'dblclick', useCapture: false },
        { type: 'input', useCapture: false },
        { type: 'submit', useCapture: false },
        { type: 'scroll', useCapture: false },
        { type: 'keydown', useCapture: false },
        { type: 'keyup', useCapture: false },
        { type: 'keypress', useCapture: false },
        { type: 'mouseover', useCapture: false },
        { type: 'change', useCapture: false },
        { type: 'focus', useCapture: true },
        { type: 'blur', useCapture: true }
      ];
      
      // 移除所有事件监听器
      eventTypes.forEach(({ type, useCapture }) => {
        if (this.eventListeners[type]) {
          iframeDoc.removeEventListener(type, this.eventListeners[type], useCapture);
        }
      });
      
      // 清空事件监听器对象
      this.eventListeners = {};
      
      // 断开MutationObserver
      if (this.mutationObserver) {
        this.mutationObserver.disconnect();
        this.mutationObserver = null;
      }
      
      addLog('已移除iframe事件监听器', 'info');
    } catch (e) {
      addLog('移除iframe事件监听器失败: ' + (e as Error).message, 'error');
    }
  }

  // 监听iframe内部的导航事件
  monitorNavigation() {
    if (!this.domViewRef || !this.domViewRef.contentWindow) {
      addLog('无法访问iframe内容窗口', 'error');
      return;
    }

    try {
      // 监听iframe内部的beforeunload事件
      const beforeUnloadHandler = () => {
        addLog('iframe即将导航到新页面', 'info');
        // 在页面卸载前移除事件监听器
        this.removeEventListeners();
      };
      
      this.domViewRef.contentWindow.addEventListener('beforeunload', beforeUnloadHandler as EventListener);
      
      // 监听iframe内部的popstate事件（浏览器前进后退）
      const popStateHandler = () => {
        addLog('iframe历史状态改变', 'info');
      };
      
      this.domViewRef.contentWindow.addEventListener('popstate', popStateHandler as EventListener);
      
      // 监听hashchange事件
      const hashChangeHandler = () => {
        addLog('iframe哈希值改变', 'info');
      };
      
      this.domViewRef.contentWindow.addEventListener('hashchange', hashChangeHandler as EventListener);
      
      addLog('iframe导航事件监听器设置完成', 'info');
    } catch (e) {
      addLog('设置iframe导航监听失败: ' + (e as Error).message, 'error');
    }
  }
}