package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 聊天服务错误处理工具类
 * 统一处理聊天过程中的各种异常情况
 * 委托给ErrorHandlerService进行实际处理
 */
@Slf4j
@Component
public class ChatErrorHandler {
    
    @Autowired
    private ErrorHandlerService unifiedErrorHandlerService;
    
    /**
     * 处理聊天过程中的异常
     * 
     * @param emitter SSE发射器
     * @param errorMessage 错误信息
     * @param exception 异常对象
     * @param processorType 处理器类型（可选）
     */
    public void handleChatError(SseEmitter emitter, String errorMessage, Exception exception, String processorType) {
        unifiedErrorHandlerService.handleChatError(emitter, errorMessage, exception, processorType);
    }
    
    /**
     * 处理聊天过程中的异常（）
     * 
     * @param emitter SSE发射器
     * @param errorMessage 错误信息
     */
    public void handleChatError(SseEmitter emitter, String errorMessage) {
        unifiedErrorHandlerService.handleChatError(emitter, errorMessage);
    }
    
    /**
     * 处理Token处理过程中的异常
     * 
     * @param emitter SSE发射器
     * @param processorType 处理器类型
     * @param exception 异常对象
     * @param isCompleted 完成状态标记
     */
    public void handleTokenError(SseEmitter emitter, String processorType, Exception exception, AtomicBoolean isCompleted) {
        unifiedErrorHandlerService.handleTokenError(emitter, processorType, exception, isCompleted);
    }
    
    /**
     * 处理完成回调过程中的异常
     * 
     * @param emitter SSE发射器
     * @param exception 异常对象
     */
    public void handleCompletionError(SseEmitter emitter, Exception exception) {
        unifiedErrorHandlerService.handleCompletionError(emitter, exception);
    }
    
    /**
     * 处理对话记录保存过程中的异常
     * 
     * @param emitter SSE发射器
     * @param exception 异常对象
     * @param isCompleted 完成状态标记
     */
    public void handleSaveDialogueError(SseEmitter emitter, Exception exception, AtomicBoolean isCompleted) {
        unifiedErrorHandlerService.handleSaveDialogueError(emitter, exception, isCompleted);
    }
}