package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import java.util.function.Consumer;

/**
 * Agent错误处理工具类
 * 统一处理Agent处理器中的错误逻辑
 */
@Slf4j
@Component
public class AgentErrorHandler {
    
    @Autowired
    private ErrorHandlerService errorHandlerService;
    
    /**
     * 处理401未授权错误
     *
     * @param e 异常对象
     * @return 是否为401错误
     */
    public boolean isUnauthorizedError(Throwable e) {
        return errorHandlerService.isUnauthorizedError(new Exception(e));
    }
    
    /**
     * 处理流式处理中的错误
     *
     * @param e 异常对象
     * @param tokenConsumer token处理回调函数
     * @param errorMessagePrefix 错误消息前缀
     */
    public void handleStreamError(Throwable e, Consumer<String> tokenConsumer, String errorMessagePrefix) {
        errorHandlerService.handleStreamError(e, tokenConsumer, errorMessagePrefix);
    }
    
    /**
     * 处理同步处理中的错误
     *
     * @param e 异常对象
     * @param errorMessagePrefix 错误消息前缀
     * @return 错误消息
     */
    public String handleSyncError(Throwable e, String errorMessagePrefix) {
        // 检查是否是401 Unauthorized错误
        if (isUnauthorizedError(e)) {
            log.error("LLM返回401未授权错误: {}", e.getMessage());
            return "请配置API密钥";
        } else {
            String errorMessage = e.getMessage();
            if (errorMessage == null || errorMessage.isEmpty()) {
                errorMessage = "未知错误";
            }
            return errorMessagePrefix + ": " + errorMessage;
        }
    }
    
    /**
     * 发送错误信息给客户端
     *
     * @param tokenConsumer token处理回调函数
     * @param errorMessage 错误消息
     */
    public void sendErrorMessage(Consumer<String> tokenConsumer, String errorMessage) {
        errorHandlerService.sendErrorMessage(tokenConsumer, errorMessage);
    }
    
    /**
     * 确保在异常情况下也调用完成回调
     *
     * @param tokenConsumer token处理回调函数
     * @param errorMessage 错误消息
     */
    public void ensureCompletionCallback(Consumer<String> tokenConsumer, String errorMessage) {
        if (tokenConsumer instanceof TokenConsumerWithCompletion) {
            try {
                ((TokenConsumerWithCompletion) tokenConsumer).onComplete(errorMessage);
            } catch (Exception ex) {
                log.error("调用onComplete时发生错误: {}", ex.getMessage(), ex);
            }
        }
    }
}