package pangea.hiagent.tool.impl;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.tool.annotation.Tool;
import org.springframework.stereotype.Component;

import java.util.regex.Pattern;

/**
 * 技术代码解释工具
 * 用于分析和解释技术代码的功能和实现逻辑
 */
@Slf4j
@Component
public class TechnicalCodeExplanationTool {
    
    // 代码语言模式
    private static final Pattern JAVA_PATTERN = Pattern.compile("\\b(public|private|protected|class|interface|extends|implements|import|package)\\b");
    private static final Pattern PYTHON_PATTERN = Pattern.compile("\\b(def|class|import|from|if|elif|else|for|while|try|except|finally|with|as)\\b");
    private static final Pattern JAVASCRIPT_PATTERN = Pattern.compile("\\b(function|var|let|const|if|else|for|while|try|catch|finally|class|import|export)\\b");
    
    public TechnicalCodeExplanationTool() {
        // 默认构造器
    }
    
    /**
     * 解释代码功能
     * @param code 代码内容
     * @param language 编程语言（可选）
     * @return 代码功能解释
     */
    @Tool(description = "分析并解释代码的功能和实现逻辑，支持多种编程语言")
    public String explainCode(String code, String language) {
        log.debug("开始解释代码，语言: {}", language);
        
        try {
            if (code == null || code.trim().isEmpty()) {
                log.warn("代码内容不能为空");
                return "错误：代码内容不能为空";
            }
            
            // 如果未指定语言，自动检测
            if (language == null || language.trim().isEmpty()) {
                language = detectLanguage(code);
            }
            
            // 生成代码解释
            String explanation = generateCodeExplanation(code, language);
            
            log.info("代码解释完成，语言: {}", language);
            return explanation;
        } catch (Exception e) {
            log.error("解释代码时发生错误: {}", e.getMessage(), e);
            return "解释代码时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 检测代码语言
     * @param code 代码内容
     * @return 检测到的语言
     */
    private String detectLanguage(String code) {
        // 统计各种语言关键字出现次数
        int javaScore = countMatches(code, JAVA_PATTERN);
        int pythonScore = countMatches(code, PYTHON_PATTERN);
        int javascriptScore = countMatches(code, JAVASCRIPT_PATTERN);
        
        // 返回得分最高的语言
        if (javaScore >= pythonScore && javaScore >= javascriptScore) {
            return "Java";
        } else if (pythonScore >= javascriptScore) {
            return "Python";
        } else {
            return "JavaScript";
        }
    }
    
    /**
     * 统计正则表达式匹配次数
     * @param text 文本内容
     * @param pattern 正则表达式
     * @return 匹配次数
     */
    private int countMatches(String text, Pattern pattern) {
        java.util.regex.Matcher matcher = pattern.matcher(text);
        int count = 0;
        while (matcher.find()) {
            count++;
        }
        return count;
    }
    
    /**
     * 生成代码解释
     * @param code 代码内容
     * @param language 编程语言
     * @return 代码解释
     */
    private String generateCodeExplanation(String code, String language) {
        StringBuilder explanation = new StringBuilder();
        explanation.append("代码解释 (").append(language).append("):\n\n");
        
        // 根据语言类型提供不同的解释
        switch (language.toLowerCase()) {
            case "java":
                explanation.append(explainJavaCode(code));
                break;
            case "python":
                explanation.append(explainPythonCode(code));
                break;
            case "javascript":
                explanation.append(explainJavascriptCode(code));
                break;
            default:
                explanation.append("这是一个").append(language).append("代码片段。\n");
                explanation.append("代码功能分析：\n");
                explanation.append(codeAnalysis(code));
                break;
        }
        
        return explanation.toString();
    }
    
    /**
     * 解释Java代码
     * @param code Java代码
     * @return 解释内容
     */
    private String explainJavaCode(String code) {
        StringBuilder explanation = new StringBuilder();
        explanation.append("Java代码功能分析：\n");
        
        // 检查是否包含类定义
        if (code.contains("class ")) {
            explanation.append("- 包含类定义\n");
        }
        
        // 检查是否包含方法定义
        if (code.contains("public ") || code.contains("private ") || code.contains("protected ")) {
            explanation.append("- 包含方法定义\n");
        }
        
        // 检查是否包含导入语句
        if (code.contains("import ")) {
            explanation.append("- 包含导入语句\n");
        }
        
        // 检查是否包含控制结构
        if (code.contains("if ") || code.contains("else ") || code.contains("for ") || code.contains("while ")) {
            explanation.append("- 包含控制结构\n");
        }
        
        // 检查是否包含异常处理
        if (code.contains("try ") || code.contains("catch ") || code.contains("finally ")) {
            explanation.append("- 包含异常处理\n");
        }
        
        explanation.append("\n").append(codeAnalysis(code));
        return explanation.toString();
    }
    
    /**
     * 解释Python代码
     * @param code Python代码
     * @return 解释内容
     */
    private String explainPythonCode(String code) {
        StringBuilder explanation = new StringBuilder();
        explanation.append("Python代码功能分析：\n");
        
        // 检查是否包含函数定义
        if (code.contains("def ")) {
            explanation.append("- 包含函数定义\n");
        }
        
        // 检查是否包含类定义
        if (code.contains("class ")) {
            explanation.append("- 包含类定义\n");
        }
        
        // 检查是否包含导入语句
        if (code.contains("import ") || code.contains("from ")) {
            explanation.append("- 包含导入语句\n");
        }
        
        // 检查是否包含控制结构
        if (code.contains("if ") || code.contains("elif ") || code.contains("else ") || 
            code.contains("for ") || code.contains("while ")) {
            explanation.append("- 包含控制结构\n");
        }
        
        // 检查是否包含异常处理
        if (code.contains("try:") || code.contains("except:") || code.contains("finally:")) {
            explanation.append("- 包含异常处理\n");
        }
        
        explanation.append("\n").append(codeAnalysis(code));
        return explanation.toString();
    }
    
    /**
     * 解释JavaScript代码
     * @param code JavaScript代码
     * @return 解释内容
     */
    private String explainJavascriptCode(String code) {
        StringBuilder explanation = new StringBuilder();
        explanation.append("JavaScript代码功能分析：\n");
        
        // 检查是否包含函数定义
        if (code.contains("function ") || code.contains("=>")) {
            explanation.append("- 包含函数定义\n");
        }
        
        // 检查是否包含类定义
        if (code.contains("class ")) {
            explanation.append("- 包含类定义\n");
        }
        
        // 检查是否包含导入导出语句
        if (code.contains("import ") || code.contains("export ")) {
            explanation.append("- 包含模块导入/导出\n");
        }
        
        // 检查是否包含控制结构
        if (code.contains("if ") || code.contains("else ") || code.contains("for ") || code.contains("while ")) {
            explanation.append("- 包含控制结构\n");
        }
        
        // 检查是否包含异常处理
        if (code.contains("try ") || code.contains("catch ") || code.contains("finally ")) {
            explanation.append("- 包含异常处理\n");
        }
        
        explanation.append("\n").append(codeAnalysis(code));
        return explanation.toString();
    }
    
    /**
     * 通用代码分析
     * @param code 代码内容
     * @return 分析结果
     */
    private String codeAnalysis(String code) {
        StringBuilder analysis = new StringBuilder();
        analysis.append("代码概要分析：\n");
        
        // 统计行数
        String[] lines = code.split("\n");
        analysis.append("- 代码行数: ").append(lines.length).append("\n");
        
        // 统计字符数
        analysis.append("- 字符总数: ").append(code.length()).append("\n");
        
        // 检查是否包含注释
        if (code.contains("//") || code.contains("/*") || code.contains("#") || code.contains("<!--")) {
            analysis.append("- 包含注释\n");
        }
        
        // 检查代码复杂度（简单估算）
        long semicolonCount = code.chars().filter(ch -> ch == ';').count();
        long braceCount = code.chars().filter(ch -> ch == '{' || ch == '}').count();
        if (semicolonCount > 10 || braceCount > 10) {
            analysis.append("- 代码复杂度较高\n");
        } else {
            analysis.append("- 代码复杂度较低\n");
        }
        
        return analysis.toString();
    }
}