package pangea.hiagent.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.cronutils.model.CronType;
import com.cronutils.model.definition.CronDefinitionBuilder;
import com.cronutils.parser.CronParser;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import pangea.hiagent.core.AgentChatService;
import pangea.hiagent.dto.AgentRequest;
import pangea.hiagent.model.Agent;
import pangea.hiagent.model.TimerConfig;
import pangea.hiagent.model.TimerExecutionHistory;
import pangea.hiagent.repository.TimerConfigRepository;
import pangea.hiagent.repository.TimerExecutionHistoryRepository;
import pangea.hiagent.repository.PromptTemplateRepository;
import pangea.hiagent.scheduler.TimerScheduler;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

/**
 * 定时器服务类
 * 负责定时器的管理和相关业务逻辑
 */
@Slf4j
@Service
public class TimerService {

    private final TimerConfigRepository timerConfigRepository;
    private final TimerExecutionHistoryRepository timerExecutionHistoryRepository;
    private final PromptTemplateRepository promptTemplateRepository;
    private final TimerScheduler timerScheduler;
    private final AgentChatService agentChatService;
    private final AgentService agentService;
    private final ObjectMapper objectMapper;

    // Cron解析器，支持秒级精度
    private final CronParser cronParser = new CronParser(CronDefinitionBuilder.instanceDefinitionFor(CronType.QUARTZ));

    public TimerService(TimerConfigRepository timerConfigRepository,
                        TimerExecutionHistoryRepository timerExecutionHistoryRepository,
                        PromptTemplateRepository promptTemplateRepository,
                        TimerScheduler timerScheduler,
                        AgentChatService agentChatService,
                        AgentService agentService,
                        ObjectMapper objectMapper) {
        this.timerConfigRepository = timerConfigRepository;
        this.timerExecutionHistoryRepository = timerExecutionHistoryRepository;
        this.promptTemplateRepository = promptTemplateRepository;
        this.timerScheduler = timerScheduler;
        this.agentChatService = agentChatService;
        this.agentService = agentService;
        this.objectMapper = objectMapper;
    }

    /**
     * 创建定时器
     */
    @Transactional
    public TimerConfig createTimer(TimerConfig timerConfig) {
        log.info("创建定时器: {}", timerConfig.getName());
        
        // 验证必要字段
        if (timerConfig.getName() == null || timerConfig.getName().trim().isEmpty()) {
            throw new IllegalArgumentException("定时器名称不能为空");
        }
        
        if (timerConfig.getCronExpression() == null || timerConfig.getCronExpression().trim().isEmpty()) {
            throw new IllegalArgumentException("Cron表达式不能为空");
        }
        
        if (timerConfig.getAgentId() == null || timerConfig.getAgentId().trim().isEmpty()) {
            throw new IllegalArgumentException("关联Agent ID不能为空");
        }

        // 验证Cron表达式
        validateCronExpression(timerConfig.getCronExpression());

        // 设置默认值
        if (timerConfig.getEnabled() == null) {
            timerConfig.setEnabled(0); // 默认禁用
        }

        // 计算下次执行时间
        LocalDateTime nextExecutionTime = timerScheduler.calculateNextExecutionTime(timerConfig.getCronExpression());
        timerConfig.setNextExecutionTime(nextExecutionTime);

        // 保存定时器配置到数据库
        try {
            timerConfigRepository.insert(timerConfig);
            log.info("定时器创建成功，ID: {}", timerConfig.getId());
        } catch (Exception e) {
            log.error("定时器保存到数据库失败: {}", e.getMessage(), e);
            throw new RuntimeException("定时器保存失败: " + e.getMessage(), e);
        }

        // 添加到Quartz调度器
        try {
            timerScheduler.addOrUpdateTimer(timerConfig);
        } catch (Exception e) {
            log.error("定时器添加到调度器失败: {}", e.getMessage(), e);
            // 如果调度器添加失败，回滚数据库操作
            try {
                timerConfigRepository.deleteById(timerConfig.getId());
            } catch (Exception deleteException) {
                log.error("回滚定时器数据失败: {}", deleteException.getMessage(), deleteException);
            }
            throw new RuntimeException("定时器调度配置失败: " + e.getMessage(), e);
        }

        return timerConfig;
    }

    /**
     * 更新定时器
     */
    @Transactional
    public TimerConfig updateTimer(TimerConfig timerConfig) {
        log.info("更新定时器: {}", timerConfig.getId());
        
        // 验证必要字段
        if (timerConfig.getId() == null || timerConfig.getId().trim().isEmpty()) {
            throw new IllegalArgumentException("定时器ID不能为空");
        }
        
        if (timerConfig.getName() == null || timerConfig.getName().trim().isEmpty()) {
            throw new IllegalArgumentException("定时器名称不能为空");
        }
        
        if (timerConfig.getCronExpression() == null || timerConfig.getCronExpression().trim().isEmpty()) {
            throw new IllegalArgumentException("Cron表达式不能为空");
        }
        
        if (timerConfig.getAgentId() == null || timerConfig.getAgentId().trim().isEmpty()) {
            throw new IllegalArgumentException("关联Agent ID不能为空");
        }

        // 验证Cron表达式
        validateCronExpression(timerConfig.getCronExpression());

        // 获取现有定时器
        TimerConfig existingTimer = timerConfigRepository.selectById(timerConfig.getId());
        if (existingTimer == null) {
            throw new IllegalArgumentException("定时器不存在: " + timerConfig.getId());
        }
        
        // 保留原始创建信息
        timerConfig.setCreatedBy(existingTimer.getCreatedBy());
        timerConfig.setCreatedAt(existingTimer.getCreatedAt());

        // 计算下次执行时间
        LocalDateTime nextExecutionTime = timerScheduler.calculateNextExecutionTime(timerConfig.getCronExpression());
        timerConfig.setNextExecutionTime(nextExecutionTime);

        // 更新数据库中的定时器配置
        try {
            timerConfigRepository.updateById(timerConfig);
            log.info("定时器更新成功，ID: {}", timerConfig.getId());
        } catch (Exception e) {
            log.error("定时器更新到数据库失败: {}", e.getMessage(), e);
            throw new RuntimeException("定时器更新失败: " + e.getMessage(), e);
        }

        // 更新Quartz调度器中的定时任务
        try {
            timerScheduler.addOrUpdateTimer(timerConfig);
        } catch (Exception e) {
            log.error("定时器更新调度器失败: {}", e.getMessage(), e);
            throw new RuntimeException("定时器调度更新失败: " + e.getMessage(), e);
        }

        return timerConfig;
    }

    /**
     * 删除定时器
     */
    @Transactional
    public void deleteTimer(String id) {
        log.info("删除定时器: {}", id);
        // 从Quartz调度器中删除
        timerScheduler.deleteTimer(id);
        // 从数据库中删除
        timerConfigRepository.deleteById(id);
    }

    /**
     * 启用定时器
     */
    @Transactional
    public void enableTimer(String id) {
        log.info("启用定时器: {}", id);
        TimerConfig timerConfig = timerConfigRepository.selectById(id);
        if (timerConfig != null) {
            timerConfig.setEnabled(1);
            // 更新数据库
            timerConfigRepository.updateById(timerConfig);
            // 更新Quartz调度器
            timerScheduler.enableTimer(id);
        }
    }

    /**
     * 禁用定时器
     */
    @Transactional
    public void disableTimer(String id) {
        log.info("禁用定时器: {}", id);
        TimerConfig timerConfig = timerConfigRepository.selectById(id);
        if (timerConfig != null) {
            timerConfig.setEnabled(0);
            // 更新数据库
            timerConfigRepository.updateById(timerConfig);
            // 更新Quartz调度器
            timerScheduler.disableTimer(id);
        }
    }

    /**
     * 获取定时器详情
     */
    public TimerConfig getTimerById(String id) {
        if (id == null || id.isEmpty()) {
            log.warn("尝试使用无效ID获取定时器");
            return null;
        }
        return timerConfigRepository.selectById(id);
    }

    /**
     * 获取定时器列表
     */
    public List<TimerConfig> listTimers() {
        List<TimerConfig> timers = timerConfigRepository.selectList(null);
        log.info("获取到 {} 个定时器", timers != null ? timers.size() : 0);
        return timers != null ? timers : List.of();
    }

    /**
     * 根据创建人获取定时器列表
     */
    public List<TimerConfig> listTimersByCreatedBy(String createdBy) {
        LambdaQueryWrapper<TimerConfig> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(TimerConfig::getCreatedBy, createdBy);
        return timerConfigRepository.selectList(wrapper);
    }

    /**
     * 验证Cron表达式有效性
     */
    private void validateCronExpression(String cronExpression) {
        if (cronExpression == null || cronExpression.isEmpty()) {
            throw new IllegalArgumentException("Cron表达式不能为空");
        }

        try {
            cronParser.parse(cronExpression);
        } catch (Exception e) {
            throw new IllegalArgumentException("无效的Cron表达式: " + e.getMessage());
        }
    }

    /**
     * 执行定时器任务
     * 由定时任务调度器调用
     */
    @Transactional
    public void executeTimerTask(String timerId) {
        log.info("开始执行定时器任务: {}", timerId);
        long startTime = System.currentTimeMillis();
        TimerExecutionHistory history = new TimerExecutionHistory();
        boolean historySaved = false;

        try {
            // 获取定时器配置
            TimerConfig timerConfig = timerConfigRepository.selectById(timerId);
            if (timerConfig == null) {
                log.error("定时器配置不存在: {}", timerId);
                return;
            }

            // 记录执行历史基本信息
            history.setTimerId(timerId);
            history.setTimerName(timerConfig.getName());
            history.setExecutionTime(LocalDateTime.now());
            history.setCreatedBy(timerConfig.getCreatedBy()); // 设置创建人
            history.setCreatedAt(LocalDateTime.now()); // 设置创建时间

            // 获取关联的Agent
            Agent agent = agentService.getAgent(timerConfig.getAgentId());
            if (agent == null) {
                log.error("关联的Agent不存在: {}", timerConfig.getAgentId());
                history.setSuccess(0);
                history.setErrorMessage("关联的Agent不存在: " + timerConfig.getAgentId());
                timerExecutionHistoryRepository.insert(history);
                historySaved = true;
                return;
            }

            // 解析动态参数
            Map<String, Object> params = null;
            if (timerConfig.getParamsJson() != null && !timerConfig.getParamsJson().isEmpty() && 
                !"{}".equals(timerConfig.getParamsJson()) && !"\"{}\"".equals(timerConfig.getParamsJson())) {
                try {
                    // 处理可能的转义JSON字符串情况
                    String cleanParamsJson = timerConfig.getParamsJson().trim();
                    // 如果是带引号的JSON字符串，则先去除外层引号
                    if (cleanParamsJson.startsWith("\"") && cleanParamsJson.endsWith("\"")) {
                        cleanParamsJson = cleanParamsJson.substring(1, cleanParamsJson.length() - 1).replace("\\\"", "\"");
                    }
                    
                    // 只有当JSON不是空对象且不为空字符串时才进行解析
                    if (!"{}".equals(cleanParamsJson) && !cleanParamsJson.isEmpty() && !"\"\"".equals(cleanParamsJson)) {
                        params = objectMapper.readValue(cleanParamsJson, Map.class);
                    }
                } catch (Exception e) {
                    log.error("解析参数JSON失败: {}", timerConfig.getParamsJson(), e);
                    history.setSuccess(0);
                    history.setErrorMessage("解析参数JSON失败: " + e.getMessage());
                    timerExecutionHistoryRepository.insert(history);
                    historySaved = true;
                    return;
                }
            }

            // 渲染提示词模板
            String prompt = timerConfig.getPromptTemplate();
            if (prompt != null && params != null && !params.isEmpty()) {
                prompt = renderPromptTemplate(prompt, params);
            } else if (prompt == null) {
                prompt = ""; // 确保prompt不为null
            }

            // 记录实际执行的提示词
            history.setActualPrompt(prompt);

            // 构建Agent请求
            AgentRequest agentRequest = AgentRequest.builder()
                    .agentId(agent.getId())
                    .userMessage(prompt)
                    .systemPrompt(agent.getPromptTemplate())
                    .model(agent.getDefaultModel())
                    .temperature(agent.getTemperature())
                    .maxTokens(agent.getMaxTokens())
                    .topP(agent.getTopP())
                    .build();

            // 调用Agent执行任务
            String result;
            if (agent.getEnableReAct() != null && agent.getEnableReAct()) {
                // 使用ReAct模式执行
                result = agentChatService.handleReActAgentRequest(agent, agentRequest, timerConfig.getCreatedBy());
            } else {
                // 使用普通模式执行
                result = agentChatService.handleNormalAgentRequest(agent, agentRequest, timerConfig.getCreatedBy());
            }

            // 更新执行结果
            long duration = System.currentTimeMillis() - startTime;
            history.setSuccess(1);
            history.setResult(result);
            history.setExecutionDuration(duration);

            // 更新定时器的最后执行时间和下次执行时间
            LocalDateTime lastExecutionTime = LocalDateTime.now();
            LocalDateTime nextExecutionTime = timerScheduler.calculateNextExecutionTime(timerConfig.getCronExpression());
            timerConfig.setLastExecutionTime(lastExecutionTime);
            timerConfig.setNextExecutionTime(nextExecutionTime);
            timerConfigRepository.updateById(timerConfig);

            log.info("定时器任务执行成功: {}, 耗时: {}ms", timerId, duration);

        } catch (Exception e) {
            // 记录执行失败信息
            long duration = System.currentTimeMillis() - startTime;
            history.setSuccess(0);
            history.setErrorMessage("执行失败: " + e.getMessage());
            history.setExecutionDuration(duration);
            log.error("定时器任务执行失败: {}", timerId, e);
        } finally {
            // 保存执行历史（避免重复保存）
            if (!historySaved) {
                try {
                    history.setCreatedAt(LocalDateTime.now()); // 确保创建时间被设置
                    timerExecutionHistoryRepository.insert(history);
                } catch (Exception insertException) {
                    log.error("保存定时器执行历史记录失败: {}", timerId, insertException);
                }
            }
        }
    }

    /**
     * 渲染提示词模板
     * @param template 提示词模板
     * @param params 动态参数
     * @return 渲染后的提示词
     */
    private String renderPromptTemplate(String template, Map<String, Object> params) {
        if (template == null || template.isEmpty() || params == null || params.isEmpty()) {
            return template != null ? template : "";
        }

        String result = template;
        for (Map.Entry<String, Object> entry : params.entrySet()) {
            String placeholder = "{{" + entry.getKey() + "}}";
            // 处理null值的情况
            String value = entry.getValue() != null ? String.valueOf(entry.getValue()) : "";
            result = result.replace(placeholder, value);
        }
        return result;
    }
}
