package pangea.hiagent.web.controller;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import pangea.hiagent.web.service.ToolConfigService;
import pangea.hiagent.model.ToolConfig;

import java.util.List;
import java.util.Map;

/**
 * 工具配置控制器
 * 提供参数配置的REST API
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/tool-configs")
public class ToolConfigController {
    
    @Autowired
    private ToolConfigService toolConfigService;
    
    /**
     * 获取所有工具配置
     * @return 工具配置列表
     */
    @GetMapping
    public ResponseEntity<List<ToolConfig>> getAllToolConfigs() {
        log.debug("获取所有工具配置");
        List<ToolConfig> toolConfigs = toolConfigService.getAllToolConfigs();
        return ResponseEntity.ok(toolConfigs);
    }
    
    /**
     * 根据工具名称获取参数配置
     * @param toolName 工具名称
     * @return 参数配置键值对
     */
    @GetMapping("/{toolName}")
    public ResponseEntity<Map<String, String>> getToolParams(@PathVariable String toolName) {
        log.debug("根据工具名称获取参数配置，工具名称：{}", toolName);
        Map<String, String> params = toolConfigService.getToolParams(toolName);
        return ResponseEntity.ok(params);
    }
    
    /**
     * 根据工具名称和参数名称获取参数值
     * @param toolName 工具名称
     * @param paramName 参数名称
     * @return 参数值
     */
    @GetMapping("/{toolName}/{paramName}")
    public ResponseEntity<String> getParamValue(@PathVariable String toolName, @PathVariable String paramName) {
        log.debug("根据工具名称和参数名称获取参数值，工具名称：{}，参数名称：{}", toolName, paramName);
        String paramValue = toolConfigService.getParamValue(toolName, paramName);
        return ResponseEntity.ok(paramValue);
    }
    
    /**
     * 保存工具配置
     * @param toolConfig 工具配置对象
     * @return 保存后的工具配置对象
     */
    @PostMapping
    public ResponseEntity<ToolConfig> saveToolConfig(@RequestBody ToolConfig toolConfig) {
        log.debug("保存工具配置：{}", toolConfig);
        ToolConfig savedConfig = toolConfigService.saveToolConfig(toolConfig);
        if (savedConfig != null) {
            return ResponseEntity.ok(savedConfig);
        } else {
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).build();
        }
    }
    
    /**
     * 保存参数值
     * @param toolName 工具名称
     * @param paramName 参数名称
     * @param paramValue 参数值
     * @return 保存结果
     */
    @PutMapping("/{toolName}/{paramName}")
    public ResponseEntity<Void> saveParamValue(@PathVariable String toolName, @PathVariable String paramName, @RequestBody String paramValue) {
        log.debug("保存参数值，工具名称：{}，参数名称：{}，参数值：{}", toolName, paramName, paramValue);
        toolConfigService.saveParamValue(toolName, paramName, paramValue);
        return ResponseEntity.ok().build();
    }
    
    /**
     * 删除工具配置
     * @param id 配置ID
     * @return 删除结果
     */
    @DeleteMapping("/{id}")
    public ResponseEntity<Void> deleteToolConfig(@PathVariable String id) {
        log.debug("删除工具配置，ID：{}", id);
        toolConfigService.deleteToolConfig(id);
        return ResponseEntity.ok().build();
    }
}