package pangea.hiagent.scheduler;

import com.cronutils.model.CronType;
import com.cronutils.model.definition.CronDefinitionBuilder;
import com.cronutils.parser.CronParser;
import lombok.extern.slf4j.Slf4j;
import org.quartz.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.TimerConfig;

import java.time.LocalDateTime;

/**
 * 定时器调度管理器
 * 负责管理Quartz的Job和Trigger，实现动态添加、更新和删除定时任务
 */
@Slf4j
@Component
public class TimerScheduler {

    @Autowired
    private Scheduler scheduler;

    // Cron解析器，用于验证Cron表达式
    private final CronParser cronParser = new CronParser(CronDefinitionBuilder.instanceDefinitionFor(CronType.QUARTZ));

    /**
     * 添加或更新定时器任务
     * @param timerConfig 定时器配置信息
     */
    public void addOrUpdateTimer(TimerConfig timerConfig) {
        try {
            log.info("添加或更新定时器任务: {}", timerConfig.getId());

            // 验证Cron表达式
            cronParser.parse(timerConfig.getCronExpression());

            // 构建JobDetail
            JobDetail jobDetail = buildJobDetail(timerConfig);

            // 构建Trigger
            Trigger trigger = buildTrigger(jobDetail, timerConfig);

            // 添加或更新Job和Trigger
            if (scheduler.checkExists(jobDetail.getKey())) {
                scheduler.rescheduleJob(trigger.getKey(), trigger);
                log.info("更新定时器任务: {}", timerConfig.getId());
            } else {
                scheduler.scheduleJob(jobDetail, trigger);
                log.info("添加定时器任务: {}", timerConfig.getId());
            }

            // 如果定时器被禁用，暂停任务
            if (timerConfig.getEnabled() == 0) {
                scheduler.pauseJob(jobDetail.getKey());
                log.info("暂停定时器任务: {}", timerConfig.getId());
            } else {
                // 如果定时器被启用，恢复任务
                scheduler.resumeJob(jobDetail.getKey());
                log.info("恢复定时器任务: {}", timerConfig.getId());
            }

        } catch (Exception e) {
            log.error("添加或更新定时器任务失败: {}", timerConfig.getId(), e);
            throw new RuntimeException("添加或更新定时器任务失败: " + e.getMessage(), e);
        }
    }

    /**
     * 删除定时器任务
     * @param timerId 定时器ID
     */
    public void deleteTimer(String timerId) {
        try {
            log.info("删除定时器任务: {}", timerId);

            JobKey jobKey = JobKey.jobKey("timerJob_" + timerId, "timerGroup");
            scheduler.deleteJob(jobKey);

            log.info("删除定时器任务成功: {}", timerId);
        } catch (Exception e) {
            log.error("删除定时器任务失败: {}", timerId, e);
            throw new RuntimeException("删除定时器任务失败: " + e.getMessage(), e);
        }
    }

    /**
     * 启用定时器任务
     * @param timerId 定时器ID
     */
    public void enableTimer(String timerId) {
        try {
            log.info("启用定时器任务: {}", timerId);

            JobKey jobKey = JobKey.jobKey("timerJob_" + timerId, "timerGroup");
            scheduler.resumeJob(jobKey);

            log.info("启用定时器任务成功: {}", timerId);
        } catch (Exception e) {
            log.error("启用定时器任务失败: {}", timerId, e);
            throw new RuntimeException("启用定时器任务失败: " + e.getMessage(), e);
        }
    }

    /**
     * 禁用定时器任务
     * @param timerId 定时器ID
     */
    public void disableTimer(String timerId) {
        try {
            log.info("禁用定时器任务: {}", timerId);

            JobKey jobKey = JobKey.jobKey("timerJob_" + timerId, "timerGroup");
            scheduler.pauseJob(jobKey);

            log.info("禁用定时器任务成功: {}", timerId);
        } catch (Exception e) {
            log.error("禁用定时器任务失败: {}", timerId, e);
            throw new RuntimeException("禁用定时器任务失败: " + e.getMessage(), e);
        }
    }

    /**
     * 构建JobDetail
     * @param timerConfig 定时器配置信息
     * @return JobDetail对象
     */
    private JobDetail buildJobDetail(TimerConfig timerConfig) {
        JobDataMap jobDataMap = new JobDataMap();
        jobDataMap.put("timerId", timerConfig.getId());

        return JobBuilder.newJob(TimerJob.class)
                .withIdentity("timerJob_" + timerConfig.getId(), "timerGroup")
                .withDescription(timerConfig.getName())
                .usingJobData(jobDataMap)
                .storeDurably(false)
                .build();
    }

    /**
     * 构建Trigger
     * @param jobDetail JobDetail对象
     * @param timerConfig 定时器配置信息
     * @return Trigger对象
     */
    private Trigger buildTrigger(JobDetail jobDetail, TimerConfig timerConfig) {
        return TriggerBuilder.newTrigger()
                .forJob(jobDetail)
                .withIdentity("timerTrigger_" + timerConfig.getId(), "timerGroup")
                .withDescription(timerConfig.getName())
                .withSchedule(CronScheduleBuilder.cronSchedule(timerConfig.getCronExpression()))
                .startNow()
                .build();
    }

    /**
     * 计算下次执行时间
     * @param cronExpression Cron表达式
     * @return 下次执行时间
     */
    public LocalDateTime calculateNextExecutionTime(String cronExpression) {
        try {
            // 实现，暂时不计算下次执行时间
            // 后续可以使用其他方式实现，或者升级cron-utils库版本
            return null;
        } catch (Exception e) {
            log.error("计算下次执行时间失败: {}", cronExpression, e);
            return null;
        }
    }
}
