package pangea.hiagent.agent.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.agent.processor.AgentProcessor;
import pangea.hiagent.agent.sse.UserSseService;
import pangea.hiagent.workpanel.event.EventService;
import pangea.hiagent.model.Agent;
import pangea.hiagent.common.utils.LogUtils;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 流式请求服务
 * 负责处理流式请求
 */
@Slf4j
@Service
public class StreamRequestService {
    
    @Autowired
    private UserSseService unifiedSseService;
    
    @Autowired
    private EventService eventService;
    
    @Autowired
    private CompletionHandlerService completionHandlerService;
    
    /**
     * 处理流式请求
     * 
     * @param emitter SSE发射器
     * @param processor Agent处理器
     * @param request Agent请求
     * @param agent Agent对象
     * @param userId 用户ID
     */
    public void handleStreamRequest(SseEmitter emitter, AgentProcessor processor, pangea.hiagent.web.dto.AgentRequest request, Agent agent, String userId) {
        LogUtils.enterMethod("handleStreamRequest", emitter, processor, request, agent, userId);
        
        // 参数验证
        if (!validateParameters(emitter, processor, request, agent, userId)) {
            return;
        }
        
        // 创建流式处理的Token消费者
        StreamTokenConsumer tokenConsumer = new StreamTokenConsumer(emitter, processor, unifiedSseService, eventService, completionHandlerService);
        // 设置上下文信息，用于保存对话记录
        tokenConsumer.setContext(agent, request, userId);
        
        // 处理流式请求，将token缓冲和事件发送完全交给处理器实现
        processor.processStreamRequest(request, agent, userId, tokenConsumer);
        LogUtils.exitMethod("handleStreamRequest", "处理完成");
    }
    
    /**
     * 验证所有必需参数
     * 
     * @param emitter SSE发射器
     * @param processor Agent处理器
     * @param request Agent请求
     * @param agent Agent对象
     * @param userId 用户ID
     * @return 验证是否通过
     */
    private boolean validateParameters(SseEmitter emitter, AgentProcessor processor, pangea.hiagent.web.dto.AgentRequest request, Agent agent, String userId) {
        return emitter != null && processor != null && request != null && agent != null && userId != null && !userId.isEmpty();
    }
    
    /**
     * 流式处理的Token消费者实现
     * 用于处理来自Agent处理器的token流，并将其转发给SSE emitter
     */
    public static class StreamTokenConsumer implements TokenConsumerWithCompletion {
        private final SseEmitter emitter;
        private final AgentProcessor processor;
        private final EventService eventService;
        private final AtomicBoolean isCompleted = new AtomicBoolean(false);

        private Agent agent;
        private pangea.hiagent.web.dto.AgentRequest request;
        private String userId;
        private CompletionHandlerService completionHandlerService;
        
        public StreamTokenConsumer(SseEmitter emitter, AgentProcessor processor, UserSseService unifiedSseService, EventService eventService, CompletionHandlerService completionHandlerService) {
            this.emitter = emitter;
            this.processor = processor;
            this.eventService = eventService;
            this.completionHandlerService = completionHandlerService;
        }
        
        public void setContext(Agent agent, pangea.hiagent.web.dto.AgentRequest request, String userId) {
            this.agent = agent;
            this.request = request;
            this.userId = userId;
        }

        @Override
        public void accept(String token) {
            // 使用JSON格式发送token，确保转义序列被正确处理
            try {
                if (!isCompleted.get()) {
                    // 检查是否是错误消息（以[错误]或[ERROR]开头）
                    if (token != null && (token.startsWith("[错误]") || token.startsWith("[ERROR]"))) {
                        // 发送标准错误事件而不是纯文本
                        eventService.sendErrorEvent(emitter, token);
                    } else {
                        // 使用SSE标准事件格式发送token，以JSON格式确保转义序列正确处理
                        eventService.sendTokenEvent(emitter, token);
                    }
                }
            } catch (Exception e) {
                log.error("发送token失败", e);
            }
        }

        @Override
        public void onComplete(String fullContent) {
            // 处理完成时的回调
            if (isCompleted.getAndSet(true)) {
                log.debug("{} Agent处理已完成，跳过重复的完成回调", processor.getProcessorType());
                return;
            }

            log.info("{} Agent处理完成，总字符数: {}", processor.getProcessorType(), fullContent != null ? fullContent.length() : 0);

            try {
                // 使用CompletionHandlerService处理完成回调
                if (completionHandlerService != null) {
                    completionHandlerService.handleCompletion(emitter, processor, agent, request, userId, fullContent, isCompleted);
                } else {
                    // 如果completionHandlerService不可用，使用默认处理逻辑
                    try {
                        // 发送完成事件
                        emitter.send("[DONE]");
                        
                        // 完成 emitter
                        emitter.complete();
                    } catch (Exception e) {
                        log.error("处理完成事件失败", e);
                    }                }
            } catch (Exception e) {
                log.error("处理完成事件失败", e);
                // 确保即使出现异常也完成emitter
                try {
                    emitter.complete();
                } catch (Exception ex) {
                    log.error("完成emitter时发生错误", ex);
                }
            }
        }    }
}