package pangea.hiagent.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import pangea.hiagent.config.AppConfig;
import pangea.hiagent.model.LlmConfig;
import pangea.hiagent.repository.LlmConfigRepository;

import java.util.List;

/**
 * LLM配置服务类
 * 负责LLM配置的管理和相关业务逻辑
 */
@Slf4j
@Service
public class LlmConfigService {
    
    private final LlmConfigRepository llmConfigRepository;
    private final Environment environment;
    private final AppConfig appConfig;
    
    public LlmConfigService(LlmConfigRepository llmConfigRepository, Environment environment, AppConfig appConfig) {
        this.llmConfigRepository = llmConfigRepository;
        this.environment = environment;
        this.appConfig = appConfig;
        // 初始化默认配置
        initializeDefaultConfigs();
    }
    
    /**
     * 初始化默认配置
     */
    private void initializeDefaultConfigs() {
        // 检查是否已有配置，如果没有则创建默认配置
        if (llmConfigRepository.selectCount(null) == 0) {
            log.info("初始化默认LLM配置");
            
            // 创建默认的DeepSeek配置，从环境变量获取API密钥
            String deepseekApiKey = environment.getProperty("DEEPSEEK_API_KEY", 
                    appConfig.getLlm().getDeepseek().getDefaultApiKey());
            LlmConfig deepseekConfig = LlmConfig.builder()
                    .name("deepseek-default")
                    .description("DeepSeek默认配置")
                    .provider("deepseek")
                    .modelName(appConfig.getLlm().getDeepseek().getDefaultModel())
                    .apiKey(deepseekApiKey)
                    .baseUrl(appConfig.getLlm().getDeepseek().getBaseUrl())
                    .temperature(appConfig.getAgent().getDefaultTemperature())
                    .maxTokens(appConfig.getAgent().getDefaultMaxTokens())
                    .topP(0.9)
                    .enabled(true) // 总是启用，即使没有API密钥也能正常启动
                    .owner("system")
                    .build();
            llmConfigRepository.insert(deepseekConfig);
            
            // 创建默认的OpenAI配置
            String openaiApiKey = environment.getProperty("OPENAI_API_KEY", 
                    appConfig.getLlm().getOpenai().getDefaultApiKey());
            LlmConfig openaiConfig = LlmConfig.builder()
                    .name("openai-default")
                    .description("OpenAI默认配置")
                    .provider("openai")
                    .modelName(appConfig.getLlm().getOpenai().getDefaultModel())
                    .apiKey(openaiApiKey)
                    .baseUrl(appConfig.getLlm().getOpenai().getBaseUrl())
                    .temperature(appConfig.getAgent().getDefaultTemperature())
                    .maxTokens(appConfig.getAgent().getDefaultMaxTokens())
                    .topP(0.9)
                    .enabled(!openaiApiKey.isEmpty()) // 仅在配置了API密钥时启用
                    .owner("system")
                    .build();
            llmConfigRepository.insert(openaiConfig);
            
            // 创建默认的Ollama配置
            LlmConfig ollamaConfig = LlmConfig.builder()
                    .name("ollama-default")
                    .description("Ollama默认配置")
                    .provider("ollama")
                    .modelName(appConfig.getLlm().getOllama().getDefaultModel())
                    .apiKey("")
                    .baseUrl(appConfig.getLlm().getOllama().getBaseUrl())
                    .temperature(appConfig.getAgent().getDefaultTemperature())
                    .maxTokens(appConfig.getAgent().getDefaultMaxTokens())
                    .topP(0.9)
                    .enabled(true)
                    .owner("system")
                    .build();
            llmConfigRepository.insert(ollamaConfig);
        } else {
            // 如果已有配置，检查是否有启用的配置，如果没有则启用所有配置
            List<LlmConfig> allConfigs = llmConfigRepository.selectList(null);
            boolean hasEnabledConfig = allConfigs.stream().anyMatch(LlmConfig::getEnabled);
            
            if (!hasEnabledConfig) {
                log.info("未发现启用的LLM配置，正在启用所有现有配置");
                for (LlmConfig config : allConfigs) {
                    config.setEnabled(true);
                    llmConfigRepository.updateById(config);
                }
            }
        }
    }
    
    /**
     * 创建LLM配置
     */
    @Transactional
    public LlmConfig createLlmConfig(LlmConfig config) {
        log.info("创建LLM配置: {}", config.getName());
        llmConfigRepository.insert(config);
        return config;
    }
    
    /**
     * 更新LLM配置
     */
    @Transactional
    public LlmConfig updateLlmConfig(LlmConfig config) {
        log.info("更新LLM配置: {}", config.getName());
        llmConfigRepository.updateById(config);
        return config;
    }
    
    /**
     * 删除LLM配置
     */
    @Transactional
    public void deleteLlmConfig(String id) {
        log.info("删除LLM配置: {}", id);
        llmConfigRepository.deleteById(id);
    }
    
    /**
     * 获取LLM配置详情
     * 
     * @param id 配置ID
     * @return LLM配置对象，如果不存在则返回null
     */
    public LlmConfig getLlmConfig(String id) {
        log.info("获取LLM配置，ID: {}", id);
        
        if (id == null || id.isEmpty()) {
            log.warn("尝试使用无效ID获取LLM配置");
            return null;
        }
        
        LlmConfig config = llmConfigRepository.selectById(id);
        log.info("获取到LLM配置: {}", config);
        return config;
    }
    
    /**
     * 根据名称获取LLM配置
     * 
     * @param name 配置名称
     * @return LLM配置对象，如果不存在则返回null
     */
    public LlmConfig getLlmConfigByName(String name) {
        if (name == null || name.isEmpty()) {
            log.warn("尝试使用无效名称获取LLM配置");
            return null;
        }
        LambdaQueryWrapper<LlmConfig> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(LlmConfig::getName, name);
        return llmConfigRepository.selectOne(wrapper);
    }
    
    /**
     * 获取启用的LLM配置列表
     */
    public List<LlmConfig> getEnabledLlmConfigs() {
        LambdaQueryWrapper<LlmConfig> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(LlmConfig::getEnabled, true);
        return llmConfigRepository.selectList(wrapper);
    }
    
    /**
     * 获取LLM配置列表
     */
    public List<LlmConfig> listLlmConfigs() {
        return llmConfigRepository.selectList(null);
    }
    
    /**
     * 分页获取LLM配置列表
     */
    public IPage<LlmConfig> pageLlmConfigs(Long current, Long size, String name, String provider) {
        Page<LlmConfig> page = new Page<>(current, size);
        
        LambdaQueryWrapper<LlmConfig> wrapper = new LambdaQueryWrapper<>();
        if (name != null && !name.isEmpty()) {
            wrapper.like(LlmConfig::getName, name);
        }
        if (provider != null && !provider.isEmpty()) {
            wrapper.eq(LlmConfig::getProvider, provider);
        }
        
        return llmConfigRepository.selectPage(page, wrapper);
    }
}