/*
 * @Description: 表单页store
 * @Author: gaorubin
 * @Date: 2025-12-24 16:12:58
 * @LastEditors: gaorubin
 * @LastEditTime: 2025-12-24 17:26:54
 */
import { defineStore } from "pinia";
import { cloneDeep } from "lodash-es";

/**
 * 获取初始表单 JSON 配置
 * 使用函数返回新对象，避免共享引用
 */
const getInitFormJson = () => ({
  pages: [
    {
      key: 0,
      type: "default",
      name: "默认页",
      code: "",
      display: "",
      props: {
        margin: "16px",
        padding: "12px",
        backgroundColor: "white",
        display: {},
      },
      bindProps: {},
      coms: [
        {
          key: 1,
          type: "node",
          name: "表单容器",
          code: "HiFormContainer",
          display: "",
          props: {
            status: "default",
            backgroundColor: "transparent",
            layout: "horizontal",
            size: "medium",
            labelAlign: "right",
            display: {},
            borderRadius: {},
            boxShadow: {},
            loop: {
              data: [],
            },
          },
          bindProps: {},
          coms: [],
        },
      ],
    },
  ],
  params: [],
  apis: [],
  funcs: [],
  pageTemplate: {},
});

export const useFormStore = defineStore("form", {
  state: () => ({
    // 表单 JSON 配置
    formJson: getInitFormJson(),

    // 是否显示表单页面
    showForm: false,

    // 表单提交回调函数
    onSubmitCallback: null as ((data: any) => void) | null,

    // 过滤掉的隐藏字段
    hiddenFields: [] as Array<{ title: string; name: string }>,
  }),

  actions: {
    /**
     * 设置表单 JSON 配置
     * @param json 表单配置对象
     */
    setFormJson(json: any) {
      this.formJson = json;
    },

    /**
     * 清空表单 JSON 配置
     */
    clearFormJson() {
      this.formJson = getInitFormJson();
    },

    /**
     * 设置隐藏字段
     * @param fields 隐藏字段数组
     */
    setHiddenFields(fields: Array<{ title: string; name: string }>) {
      this.hiddenFields = fields;
    },

    /**
     * 清空隐藏字段
     */
    clearHiddenFields() {
      this.hiddenFields = [];
    },

    /**
     * 设置表单提交回调函数
     * @param callback 回调函数
     */
    setSubmitCallback(callback: (data: any) => void) {
      this.onSubmitCallback = callback;
    },

    /**
     * 触发表单提交
     * @param data 表单数据
     */
    submitForm(data: any) {
      if (this.onSubmitCallback) {
        this.onSubmitCallback(data);
      }
      // 提交后关闭表单
      this.closeForm();
    },

    /**
     * 显示表单页面
     * @param json 可选的表单配置对象，格式为 { coms: [] }
     */
    openForm(json?: any) {
      if (json) {
        // 验证 json 的有效性
        if (!json.coms || !Array.isArray(json.coms)) {
          console.error("无效的表单配置：缺少 coms 数组", json);
          return;
        }

        // 获取初始表单配置
        const formConfig = getInitFormJson();

        // 将接收到的 coms 放入表单容器的 coms 数组中
        // 表单容器位于 pages[0].coms[0]（HiFormContainer）
        if (
          formConfig.pages &&
          formConfig.pages[0] &&
          formConfig.pages[0].coms &&
          formConfig.pages[0].coms[0]
        ) {
          formConfig.pages[0].coms[0].coms = json.coms;
        } else {
          console.error("INIT_FORM_JSON 结构异常，无法找到表单容器");
          return;
        }

        // 设置表单配置
        this.setFormJson(formConfig);

        this.showForm = true;
      }
    },

    /**
     * 隐藏表单页面
     */
    closeForm() {
      this.showForm = false;
      this.clearFormJson();
      this.clearHiddenFields();
    },
  },
});
