package pangea.hiagent.react;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.workpanel.IWorkPanelDataCollector;

/**
 * 自定义 ReAct 回调类，用于捕获并处理 ReAct 的每一步思维过程
 * 适配项目现有的 ReAct 实现方式
 */
@Slf4j
@Component // 注册为 Spring 组件，方便注入
public class DefaultReactCallback implements ReactCallback {
    
    @Autowired
    private IWorkPanelDataCollector workPanelCollector;

    /**
     * ReAct 每执行一个步骤，该方法会被触发
     * @param reactStep ReAct 步骤对象，包含步骤的所有核心信息
     */
    @Override
    public void onStep(ReactStep reactStep) {
        // 将信息记录到工作面板
        recordReactStepToWorkPanel(reactStep);
    }
    
    /**
     * 处理 ReAct 最终答案步骤
     * @param finalAnswer 最终答案
     */
    @Override
    public void onFinalAnswer(String finalAnswer) {
        // 创建一个FINAL_ANSWER类型的ReactStep并处理
        ReactStep finalStep = new ReactStep(0, ReactStepType.FINAL_ANSWER, finalAnswer);
        recordReactStepToWorkPanel(finalStep);
    }
    
    /**
     * 将ReAct步骤记录到工作面板
     * @param reactStep ReAct步骤
     */
    private void recordReactStepToWorkPanel(ReactStep reactStep) {
        if (workPanelCollector == null) {
            log.debug("无法记录到工作面板：collector为null");
            return;
        }
        
        try {
            switch (reactStep.getStepType()) {
                case THOUGHT:
                    workPanelCollector.recordThinking(reactStep.getContent(), "reasoning");
                    break;
                case ACTION:
                    if (reactStep.getAction() != null) {
                        workPanelCollector.recordToolCallStart(
                            reactStep.getAction().getToolName(), 
                            "execute", 
                            reactStep.getAction().getParameters()
                        );
                    }
                    break;
                case OBSERVATION:
                    if (reactStep.getObservation() != null) {
                        workPanelCollector.recordToolCallComplete(
                            "Unknown", 
                            reactStep.getObservation().getContent(), 
                            "success"
                        );
                    }
                    break;
                case FINAL_ANSWER:
                    workPanelCollector.recordThinking(reactStep.getContent(), "final_answer");
                    break;
                default:
                    log.warn("未知的ReAct步骤类型: {}", reactStep.getStepType());
                    break;
            }
        } catch (Exception e) {
            log.error("记录ReAct步骤到工作面板失败", e);
        }
    }
}