package pangea.hiagent.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.stereotype.Service;
import pangea.hiagent.dto.TimerExecutionHistoryDto;
import pangea.hiagent.model.TimerExecutionHistory;
import pangea.hiagent.repository.TimerExecutionHistoryRepository;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 执行历史服务类
 * 负责执行历史的查询、统计和管理
 */
@Service
public class HistoryService {

    private final TimerExecutionHistoryRepository timerExecutionHistoryRepository;
    private final DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    public HistoryService(TimerExecutionHistoryRepository timerExecutionHistoryRepository) {
        this.timerExecutionHistoryRepository = timerExecutionHistoryRepository;
    }

    /**
     * 获取执行历史列表，支持多条件筛选和分页
     * 
     * @param timerId   定时器ID
     * @param success   执行结果（1-成功，0-失败）
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param page      页码
     * @param size      每页大小
     * @return 执行历史列表
     */
    public Page<TimerExecutionHistoryDto> getExecutionHistoryList(
            String timerId, Integer success, String startTime, String endTime, int page, int size) {
        // 构建查询条件
        LambdaQueryWrapper<TimerExecutionHistory> wrapper = buildQueryWrapper(timerId, success, startTime, endTime);
        
        // 按执行时间倒序排序
        wrapper.orderByDesc(TimerExecutionHistory::getExecutionTime);
        
        // 分页查询
        Page<TimerExecutionHistory> pagination = new Page<>(page, size);
        timerExecutionHistoryRepository.selectPage(pagination, wrapper);
        
        // 转换为DTO
        List<TimerExecutionHistoryDto> records = pagination.getRecords().stream()
                .map(this::convertToDto)
                .collect(Collectors.toList());
        
        // 创建新的分页对象并设置数据
        Page<TimerExecutionHistoryDto> resultPage = new Page<>(pagination.getCurrent(), pagination.getSize(), pagination.getTotal());
        resultPage.setRecords(records);
        
        return resultPage;
    }

    /**
     * 获取指定定时器的执行历史
     * 
     * @param timerId 定时器ID
     * @param page    页码
     * @param size    每页大小
     * @return 执行历史列表
     */
    public Page<TimerExecutionHistoryDto> getExecutionHistoryByTimerId(String timerId, int page, int size) {
        return getExecutionHistoryList(timerId, null, null, null, page, size);
    }

    /**
     * 获取执行历史详情
     * 
     * @param id 执行历史ID
     * @return 执行历史详情
     */
    public TimerExecutionHistoryDto getExecutionHistoryDetail(Long id) {
        TimerExecutionHistory history = timerExecutionHistoryRepository.selectById(id);
        return history != null ? convertToDto(history) : null;
    }

    /**
     * 构建查询条件
     */
    private LambdaQueryWrapper<TimerExecutionHistory> buildQueryWrapper(
            String timerId, Integer success, String startTime, String endTime) {
        LambdaQueryWrapper<TimerExecutionHistory> wrapper = new LambdaQueryWrapper<>();
        
        // 定时器ID条件
        if (timerId != null && !timerId.isEmpty()) {
            wrapper.eq(TimerExecutionHistory::getTimerId, timerId);
        }
        
        // 执行结果条件
        if (success != null) {
            wrapper.eq(TimerExecutionHistory::getSuccess, success);
        }
        
        // 开始时间条件
        if (startTime != null && !startTime.isEmpty()) {
            LocalDateTime start = LocalDateTime.parse(startTime, formatter);
            wrapper.ge(TimerExecutionHistory::getExecutionTime, start);
        }
        
        // 结束时间条件
        if (endTime != null && !endTime.isEmpty()) {
            LocalDateTime end = LocalDateTime.parse(endTime, formatter);
            wrapper.le(TimerExecutionHistory::getExecutionTime, end);
        }
        
        return wrapper;
    }

    /**
     * 转换实体为DTO
     */
    private TimerExecutionHistoryDto convertToDto(TimerExecutionHistory history) {
        return TimerExecutionHistoryDto.builder()
                .id(history.getId() != null ? history.getId().toString() : null)
                .timerId(history.getTimerId())
                .timerName(history.getTimerName())
                .executionTime(history.getExecutionTime())
                .success(history.getSuccess())
                .result(history.getResult())
                .errorMessage(history.getErrorMessage())
                .actualPrompt(history.getActualPrompt())
                .executionDuration(history.getExecutionDuration())
                .build();
    }

    /**
     * 统计定时器执行成功次数
     * 
     * @param timerId 定时器ID
     * @return 成功次数
     */
    public long countSuccessExecution(String timerId) {
        LambdaQueryWrapper<TimerExecutionHistory> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(TimerExecutionHistory::getTimerId, timerId)
                .eq(TimerExecutionHistory::getSuccess, 1);
        return timerExecutionHistoryRepository.selectCount(wrapper);
    }

    /**
     * 统计定时器执行失败次数
     * 
     * @param timerId 定时器ID
     * @return 失败次数
     */
    public long countFailedExecution(String timerId) {
        LambdaQueryWrapper<TimerExecutionHistory> wrapper = new LambdaQueryWrapper<>();
        wrapper.eq(TimerExecutionHistory::getTimerId, timerId)
                .eq(TimerExecutionHistory::getSuccess, 0);
        return timerExecutionHistoryRepository.selectCount(wrapper);
    }
}