package pangea.hiagent.memory;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.messages.Message;
import org.springframework.ai.chat.messages.UserMessage;
import org.springframework.ai.chat.messages.AssistantMessage;
import org.springframework.ai.chat.messages.SystemMessage;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * 智能历史摘要服务
 * 实现关键信息识别算法，优化长对话历史管理
 */
@Slf4j
@Service
public class SmartHistorySummarizer {

    // 关键词模式，用于识别重要信息
    private static final Pattern IMPORTANT_KEYWORDS_PATTERN = Pattern.compile(
        "(重要|关键|主要|核心|总结|结论|决定|计划|目标|问题|解决方案|步骤|方法|原因|结果|影响|建议|需求|要求|规则|限制|约束|前提|假设|定义|概念|术语|公式|代码|示例|案例|经验|教训|注意|警告|错误|异常|bug|debug|修复|优化|改进|提升|增强|加强|完善|完成|实现|达成|达到|获得|取得|收获|学习|理解|掌握|熟悉|了解|知道|明白|清楚|确认|验证|测试|检查|审查|审核|批准|同意|拒绝|反对|支持|帮助|协助|合作|配合|沟通|交流|讨论|会议|电话|邮件|联系|通知|提醒|警告|紧急|重要|优先|首要|第一|最后|最终|结束|完成|完毕)",
        Pattern.CASE_INSENSITIVE
    );

    // 代码模式，用于识别代码片段
    private static final Pattern CODE_PATTERN = Pattern.compile(
        "(```|\\b(function|class|def|public|private|protected|static|void|int|String|boolean|if|else|for|while|switch|case|break|continue|return|try|catch|finally|throw|throws|import|package|extends|implements|interface|abstract|final|native|synchronized|volatile|transient|strictfp)\\b)",
        Pattern.CASE_INSENSITIVE
    );

    /**
     * 智能摘要历史消息
     * @param messages 原始消息列表
     * @param maxLength 最大保留消息数
     * @return 摘要后的消息列表
     */
    public List<Message> summarize(List<Message> messages, int maxLength) {
        if (messages == null || messages.size() <= maxLength) {
            return messages;
        }

        log.debug("开始智能摘要历史消息，原始消息数: {}, 目标长度: {}", messages.size(), maxLength);

        // 识别重要消息
        List<Message> importantMessages = identifyImportantMessages(messages);
        
        // 如果重要消息数量小于等于目标长度，直接返回
        if (importantMessages.size() <= maxLength) {
            log.debug("重要消息数量({})小于等于目标长度({})，直接返回", importantMessages.size(), maxLength);
            return importantMessages;
        }

        // 如果重要消息仍然过多，进一步筛选
        List<Message> filteredMessages = filterMessages(importantMessages, maxLength);
        log.debug("筛选后消息数量: {}", filteredMessages.size());
        return filteredMessages;
    }

    /**
     * 识别重要消息
     * @param messages 消息列表
     * @return 重要消息列表
     */
    private List<Message> identifyImportantMessages(List<Message> messages) {
        List<Message> importantMessages = new ArrayList<>();
        
        for (Message message : messages) {
            // 始终保留系统消息
            if (message instanceof SystemMessage) {
                importantMessages.add(message);
                continue;
            }
            
            // 检查是否包含重要关键词
            if (containsImportantKeywords(message)) {
                importantMessages.add(message);
                continue;
            }
            
            // 检查是否包含代码
            if (containsCode(message)) {
                importantMessages.add(message);
                continue;
            }
            
            // 对于用户消息和助手消息，保留最近的一部分
            if ((message instanceof UserMessage || message instanceof AssistantMessage) && 
                importantMessages.size() < messages.size() * 0.7) { // 保留最多70%的普通消息
                importantMessages.add(message);
            }
        }
        
        return importantMessages;
    }

    /**
     * 检查消息是否包含重要关键词
     * @param message 消息
     * @return 是否包含重要关键词
     */
    private boolean containsImportantKeywords(Message message) {
        String content = message.getText();
        if (content == null || content.isEmpty()) {
            return false;
        }
        
        return IMPORTANT_KEYWORDS_PATTERN.matcher(content).find();
    }

    /**
     * 检查消息是否包含代码
     * @param message 消息
     * @return 是否包含代码
     */
    private boolean containsCode(Message message) {
        String content = message.getText();
        if (content == null || content.isEmpty()) {
            return false;
        }
        
        return CODE_PATTERN.matcher(content).find();
    }

    /**
     * 进一步筛选消息
     * @param messages 消息列表
     * @param maxLength 最大长度
     * @return 筛选后的消息列表
     */
    private List<Message> filterMessages(List<Message> messages, int maxLength) {
        // 如果消息数量仍然超过最大长度，保留最近的消息
        int startIndex = Math.max(0, messages.size() - maxLength);
        return new ArrayList<>(messages.subList(startIndex, messages.size()));
    }
}