package pangea.hiagent.controller;

import lombok.extern.slf4j.Slf4j;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.*;
import pangea.hiagent.dto.ApiResponse;
import pangea.hiagent.dto.PageData;
import pangea.hiagent.model.Agent;
import pangea.hiagent.service.AgentService;
import com.baomidou.mybatisplus.core.metadata.IPage;

/**
 * Agent API控制器
 * 提供Agent的増删改查功能
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/agent")
public class AgentController {
    
    // 常量定义（已删除，对话功能已移除）
    
    // 服务依赖
    private final AgentService agentService;
    
    public AgentController(AgentService agentService) {
        this.agentService = agentService;
    }
    
    // ==================== Agent管理API ====================
    
    /**
     * 创建Agent
     */
    @PostMapping
    public ApiResponse<Agent> createAgent(@RequestBody Agent agent) {
        try {
            String userId = getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }
            
            log.info("用户 {} 开始创建Agent: {}", userId, agent.getName());
            agent.setOwner(userId);
            Agent created = agentService.createAgent(agent);
            log.info("用户 {} 成功创建Agent: {} (ID: {})", userId, created.getName(), created.getId());
            return ApiResponse.success(created, "创建Agent成功");
        } catch (Exception e) {
            log.error("创建Agent失败", e);
            return ApiResponse.error(4001, "创建Agent失败: " + e.getMessage());
        }
    }
    
    /**
     * 更新Agent
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'Agent', 'write')")
    @PutMapping("/{id}")
    public ApiResponse<Agent> updateAgent(@PathVariable String id, @RequestBody Agent agent) {
        try {
            String userId = getCurrentUserId();
            log.info("用户 {} 开始更新Agent: {}", userId, id);
            agent.setId(id);
            Agent updated = agentService.updateAgent(agent);
            log.info("用户 {} 成功更新Agent: {}", userId, updated.getId());
            return ApiResponse.success(updated, "更新Agent成功");
        } catch (Exception e) {
            log.error("更新Agent失败", e);
            return ApiResponse.error(4001, "更新Agent失败: " + e.getMessage());
        }
    }
    
    /**
     * 删除Agent
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'Agent', 'delete')")
    @DeleteMapping("/{id}")
    public ApiResponse<Void> deleteAgent(@PathVariable String id) {
        try {
            String userId = getCurrentUserId();
            log.info("用户 {} 开始删除Agent: {}", userId, id);
            agentService.deleteAgent(id);
            log.info("用户 {} 成功删除Agent: {}", userId, id);
            return ApiResponse.success(null, "删除Agent成功");
        } catch (Exception e) {
            log.error("删除Agent失败", e);
            return ApiResponse.error(4001, "删除Agent失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取Agent详情
     */
    @PreAuthorize("@permissionEvaluator.hasPermission(authentication, #id, 'Agent', 'read')")
    @GetMapping("/{id}")
    public ApiResponse<Agent> getAgent(@PathVariable String id) {
        try {
            Agent agent = agentService.getAgent(id);
            if (agent == null) {
                return ApiResponse.error(4001, "Agent不存在");
            }
            return ApiResponse.success(agent);
        } catch (Exception e) {
            log.error("获取Agent详情失败", e);
            return ApiResponse.error(4001, "获取Agent详情失败: " + e.getMessage());
        }
    }
    
    /**
     * 分页获取Agent列表
     */
    @GetMapping("/list")
    public ApiResponse<PageData<Agent>> listAgents(
            @RequestParam(defaultValue = "1") Long current,
            @RequestParam(defaultValue = "10") Long size,
            @RequestParam(required = false) String name,
            @RequestParam(required = false) String status) {
        try {
            IPage<Agent> page = agentService.pageAgents(current, size, name, status);
            return ApiResponse.success(PageData.from(page));
        } catch (Exception e) {
            log.error("获取Agent列表失败", e);
            return ApiResponse.error(4001, "获取Agent列表失败: " + e.getMessage());
        }
    }
    
    /**
     * 获取用户的Agent列表
     */
    @GetMapping
    public ApiResponse<java.util.List<Agent>> getUserAgents() {
        try {
            String userId = getCurrentUserId();
            if (userId == null) {
                return ApiResponse.error(4001, "用户未认证");
            }
            java.util.List<Agent> agents = agentService.getUserAgents(userId);
            return ApiResponse.success(agents);
        } catch (Exception e) {
            log.error("获取用户Agent列表失败", e);
            return ApiResponse.error(4001, "获取用户Agent列表失败: " + e.getMessage());
        }
    }
    
    // ==================== 私有方法 ====================
    
    // ---------- 用户认证相关 ----------
    
    /**
     * 获取当前认证用户ID
     */
    private String getCurrentUserId() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        return (authentication != null && authentication.getPrincipal() != null) ? 
            (String) authentication.getPrincipal() : null;
    }
}