package pangea.hiagent.websocket;

import com.alibaba.fastjson2.JSON;
import com.microsoft.playwright.*;
import com.microsoft.playwright.options.LoadState;
import org.springframework.web.socket.*;
import org.springframework.web.socket.handler.TextWebSocketHandler;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * 优化的DOM同步WebSocket处理器
 * 针对前端iframe显示特点进行了简化和优化
 */
public class DomSyncHandler extends TextWebSocketHandler {
    // 存储连接的前端客户端（线程安全）
    private static final ConcurrentMap<WebSocketSession, String> clients = new ConcurrentHashMap<>();

    // Playwright核心实例
    private Playwright playwright;
    private Browser browser;
    private Page page;
    private BrowserContext context;
    
    // 连接数限制
    private static final int MAX_CONNECTIONS_PER_USER = 5; // 每用户最大连接数
    private static final int MAX_COMMANDS_PER_SECOND = 10; // 每秒最大指令数
    
    // 用户连接计数
    private static final ConcurrentMap<String, Integer> userConnections = new ConcurrentHashMap<>();
    
    // 指令频率限制
    private static final ConcurrentMap<String, Long> lastCommandTimes = new ConcurrentHashMap<>();
    private static final ConcurrentMap<String, Integer> commandCounts = new ConcurrentHashMap<>();
    
    // 统计信息
    private static final ConcurrentMap<String, Long> messageCounters = new ConcurrentHashMap<>();
    
    // 增加计数器
    private void incrementCounter(String counterName) {
        messageCounters.merge(counterName, 1L, Long::sum);
    }
    
    /**
     * 转义HTML内容中的特殊字符，防止JSON序列化错误
     * 
     * @param content 待转义的HTML内容
     * @return 转义后的内容
     */
    private String escapeHtmlContent(String content) {
        if (content == null || content.isEmpty()) {
            return content;
        }
        
        StringBuilder sb = new StringBuilder(content.length() * 2); // 预分配空间以提高性能
        
        for (int i = 0; i < content.length(); i++) {
            char c = content.charAt(i);
            switch (c) {
                case '"':
                    sb.append("\\\"");
                    break;
                case '\\':
                    sb.append("\\\\");
                    break;
                case '/':
                    sb.append("\\/");
                    break;
                case '\b':
                    sb.append("\\b");
                    break;
                case '\f':
                    sb.append("\\f");
                    break;
                case '\n':
                    sb.append("\\n");
                    break;
                case '\r':
                    sb.append("\\r");
                    break;
                case '\t':
                    sb.append("\\t");
                    break;
                default:
                    // 处理控制字符
                    if (c < 0x20) {
                        sb.append(String.format("\\u%04x", (int) c));
                    } else {
                        sb.append(c);
                    }
                    break;
            }
        }
        
        return sb.toString();
    }
    
    // 获取计数器值
    public static long getCounter(String counterName) {
        return messageCounters.getOrDefault(counterName, 0L);
    }
    
    // 重置计数器
    public static void resetCounter(String counterName) {
        messageCounters.put(counterName, 0L);
    }
    
    // 初始化Playwright与页面
    public DomSyncHandler() {
        initPlaywright();
        initPageListener();
    }

    /**
     * 初始化Playwright（服务器端无头模式，适配生产环境）
     */
    private void initPlaywright() {
        try {
            System.out.println("正在初始化Playwright...");
            playwright = Playwright.create();
            // 启动Chrome无头实例，添加必要参数
            browser = playwright.chromium().launch(new BrowserType.LaunchOptions()
                    .setHeadless(true) // 无头模式，无界面
                    .setArgs(java.util.Arrays.asList(
                            "--no-sandbox", // 服务器端必须，否则Chrome启动失败
                            "--disable-dev-shm-usage", // 解决Linux下/dev/shm空间不足的问题
                            "--disable-gpu", // 禁用GPU，服务器端无需
                            "--remote-allow-origins=*"))); // 允许跨域请求
            
            // 创建浏览器上下文（隔离环境）
            context = browser.newContext(new Browser.NewContextOptions()
                    .setViewportSize(1920, 1080) // 设置视口大小，与前端一致
                    .setUserAgent("Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36")); // 设置用户代理
            
            page = context.newPage();
            
            // 设置默认超时时间，避免长时间等待
            page.setDefaultTimeout(10000); // 10秒超时
            
            System.out.println("Playwright初始化成功");
        } catch (Exception e) {
            System.err.println("Playwright初始化失败: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * 初始化页面监听事件（核心：捕获DOM变化）
     * 针对iframe显示特点进行了优化，简化了监听逻辑
     */
    private void initPageListener() {
        // 初始化统计计数器
        messageCounters.put("domChanges", 0L);
        messageCounters.put("websocketMessages", 0L);
        messageCounters.put("errors", 0L);
        
        // 1. 页面加载完成后，推送完整的DOM（初始化）
        page.onLoad(page -> {
            System.out.println("页面加载完成: " + page.url());
            incrementCounter("pageLoads");
            // 发送完整的DOM内容到客户端
            sendFullDomToClientsWithRetry();
        });

        // 2. 监听DOM变化（使用MutationObserver），推送增量更新
        // 针对iframe特点优化：只监听body区域的变化
        page.evaluate("() => {\n" +
                "  // 创建MutationObserver监听DOM变化\n" +
                "  const observer = new MutationObserver((mutations) => {\n" +
                "    // 将变化的DOM节点转为字符串，发送给Playwright\n" +
                "    const changes = mutations.map(mutation => ({\n" +
                "      type: mutation.type,\n" +
                "      target: mutation.target.outerHTML || mutation.target.textContent,\n" +
                "      addedNodes: Array.from(mutation.addedNodes).map(node => node.outerHTML || node.textContent || ''),\n" +
                "      removedNodes: Array.from(mutation.removedNodes).map(node => node.outerHTML || node.textContent || ''),\n" +
                "      attributeName: mutation.attributeName,\n" +
                "      oldValue: mutation.oldValue\n" +
                "    }));\n" +
                "    // 调用Playwright的暴露函数，传递DOM变化数据\n" +
                "    window.domChanged(JSON.stringify(changes));\n" +
                "  });\n" +
                "  // 配置监听：监听body节点的添加/删除、属性变化、子节点变化\n" +
                "  observer.observe(document.body, {\n" +
                "    childList: true,\n" +
                "    attributes: true,\n" +
                "    subtree: true,\n" +
                "    characterData: true,\n" +
                "    attributeOldValue: true\n" +
                "  });\n" +
                "}");

        // 3. 暴露Playwright函数，接收前端的DOM变化数据
        page.exposeFunction("domChanged", args -> {
            try {
                if (args.length > 0 && args[0] instanceof String) {
                    String changes = (String) args[0];
                    if (changes != null && !changes.isEmpty()) {
                        incrementCounter("domChanges");
                        sendIncrementalDomToClients(changes);
                    }
                }
            } catch (Exception e) {
                String errorMsg = "处理DOM变化失败: " + e.getMessage();
                System.err.println(errorMsg);
                e.printStackTrace();
                incrementCounter("errors");
                sendErrorToClients(errorMsg);
            }
            return null;
        });

        // 4. 监听页面导航事件，导航后重新初始化
        page.onFrameNavigated(frame -> {
            System.out.println("检测到页面导航: " + frame.url());
            incrementCounter("navigations");
            // 异步处理导航完成后的DOM发送，避免阻塞
            java.util.concurrent.CompletableFuture.runAsync(() -> {
                try {
                    // 等待页面达到网络空闲状态，确保内容稳定
                    page.waitForLoadState(LoadState.NETWORKIDLE);
                    System.out.println("页面加载完成: " + frame.url());
                    // 发送更新后的DOM内容到客户端
                    sendFullDomToClientsWithRetry();
                } catch (Exception e) {
                    String errorMsg = "页面加载状态等待失败: " + e.getMessage();
                    System.err.println(errorMsg);
                    e.printStackTrace();
                    incrementCounter("errors");
                    sendErrorToClients(errorMsg);
                }
            });
        });
        
        // 5. 监听页面错误事件
        page.onPageError(error -> {
            try {
                String errorMsg = "页面错误: " + error;
                System.err.println(errorMsg);
                incrementCounter("errors");
                sendErrorToClients(errorMsg);
            } catch (Exception e) {
                System.err.println("处理页面错误事件失败: " + e.getMessage());
                e.printStackTrace();
            }
        });
    }

    /**
     * 推送完整的DOM给所有客户端（初始化时调用）
     * 针对iframe显示特点进行了优化，不再传输样式和脚本
     */
    private void sendFullDomToClients() {
        try {
            // 1. 获取页面完整DOM（包含所有节点）
            String fullDom = page.content();

            // 2. 对HTML内容进行转义处理，防止JSON序列化错误
            String escapedDom = escapeHtmlContent(fullDom);

            // 3. 封装数据（简化版，不传输样式和脚本）
            DomSyncData data = new DomSyncData(
                    "init", // 初始化类型
                    escapedDom,
                    "", // 不再传输样式
                    "", // 不再传输脚本
                    getCurrentPageUrl() // 当前页面URL
            );

            // 4. 序列化为JSON并推送
            String jsonData = JSON.toJSONString(data);
            broadcastMessage(jsonData);

        } catch (Exception e) {
            e.printStackTrace();
            sendErrorToClients("获取完整DOM失败：" + e.getMessage());
        }
    }

    /**
     * 推送完整的DOM给所有客户端（带重试机制）
     * 针对页面导航过程中的不稳定状态增加了重试机制
     */
    private void sendFullDomToClientsWithRetry() {
        sendFullDomToClientsWithRetry(3, 500); // 默认重试3次，每次间隔500毫秒
    }

    /**
     * 推送完整的DOM给所有客户端（带重试机制）
     * 
     * @param maxRetries 最大重试次数
     * @param retryDelay 重试间隔（毫秒）
     */
    private void sendFullDomToClientsWithRetry(int maxRetries, long retryDelay) {
        Exception lastException = null;
        
        for (int i = 0; i < maxRetries; i++) {
            try {
                // 1. 获取页面完整DOM（包含所有节点）
                String fullDom = page.content();

                // 2. 对HTML内容进行转义处理，防止JSON序列化错误
                String escapedDom = escapeHtmlContent(fullDom);

                // 3. 封装数据（简化版，不传输样式和脚本）
                DomSyncData data = new DomSyncData(
                        "init", // 初始化类型
                        escapedDom,
                        "", // 不再传输样式
                        "", // 不再传输脚本
                        getCurrentPageUrl() // 当前页面URL
                );

                // 4. 序列化为JSON并推送
                String jsonData = JSON.toJSONString(data);
                broadcastMessage(jsonData);
                
                // 成功发送，直接返回
                return;
            } catch (Exception e) {
                lastException = e;
                System.err.println("第" + (i+1) + "次获取完整DOM失败: " + e.getMessage());
                
                // 如果不是最后一次重试，则等待一段时间再重试
                if (i < maxRetries - 1) {
                    try {
                        Thread.sleep(retryDelay);
                    } catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                        sendErrorToClients("获取完整DOM被中断：" + ie.getMessage());
                        return;
                    }
                }
            }
        }
        
        // 所有重试都失败了
        if (lastException != null) {
            lastException.printStackTrace();
            sendErrorToClients("获取完整DOM失败（已重试" + maxRetries + "次）：" + lastException.getMessage());
        }
    }

    /**
     * 推送增量DOM变化给所有客户端（DOM更新时调用）
     * 针对iframe显示特点进行了优化
     */
    private void sendIncrementalDomToClients(String changes) {
        try {
            // 对变化数据进行转义处理
            String escapedChanges = escapeHtmlContent(changes);
            
            DomSyncData data = new DomSyncData(
                    "update", // 增量更新类型
                    escapedChanges, // DOM变化数据
                    "", // 样式无增量，空字符串
                    "", // 脚本无增量，空字符串
                    getCurrentPageUrl()
            );
            String jsonData = JSON.toJSONString(data);
            broadcastMessage(jsonData);
        } catch (Exception e) {
            e.printStackTrace();
            sendErrorToClients("推送增量DOM失败：" + e.getMessage());
        }
    }
    
    /**
     * 广播消息给所有客户端（增加消息大小检查和分片处理）
     */
    private void broadcastMessage(String message) {
        // 检查消息大小，如果过大则进行分片处理
        final int MAX_MESSAGE_SIZE = 64 * 1024; // 64KB限制
        
        try {
            if (message == null || message.isEmpty()) {
                return;
            }
            
            // 如果消息小于最大限制，直接发送
            if (message.length() <= MAX_MESSAGE_SIZE) {
                TextMessage textMessage = new TextMessage(message);
                for (WebSocketSession client : clients.keySet()) {
                    try {
                        if (client.isOpen()) {
                            client.sendMessage(textMessage);
                        }
                    } catch (Exception e) {
                        System.err.println("发送消息给客户端失败: " + e.getMessage());
                        e.printStackTrace();
                        // 发送失败时移除客户端
                        clients.remove(client);
                    }
                }
            } else {
                // 消息过大，需要分片处理
                System.out.println("消息过大，正在进行分片处理，总长度: " + message.length());
                sendFragmentedMessage(message, MAX_MESSAGE_SIZE);
            }
        } catch (Exception e) {
            System.err.println("广播消息时发生异常: " + e.getMessage());
            e.printStackTrace();
        }
    }
    
    /**
     * 分片发送大消息
     * 
     * @param message 完整消息
     * @param maxFragmentSize 每个片段的最大大小
     */
    private void sendFragmentedMessage(String message, int maxFragmentSize) {
        try {
            int totalLength = message.length();
            int fragmentCount = (int) Math.ceil((double) totalLength / maxFragmentSize);
            
            System.out.println("消息将被分为 " + fragmentCount + " 个片段发送");
            
            for (int i = 0; i < fragmentCount; i++) {
                int start = i * maxFragmentSize;
                int end = Math.min(start + maxFragmentSize, totalLength);
                String fragment = message.substring(start, end);
                
                // 创建分片消息
                DomSyncData fragmentData = new DomSyncData(
                    "fragment", // 分片类型
                    fragment, // 分片内容
                    "", // 样式
                    "", // 脚本
                    getCurrentPageUrl() + "?fragment=" + i + "&total=" + fragmentCount // URL附加分片信息
                );
                
                String jsonData = JSON.toJSONString(fragmentData);
                TextMessage textMessage = new TextMessage(jsonData);
                
                // 发送分片消息
                for (WebSocketSession client : clients.keySet()) {
                    try {
                        if (client.isOpen()) {
                            client.sendMessage(textMessage);
                        }
                    } catch (Exception e) {
                        System.err.println("发送分片消息给客户端失败: " + e.getMessage());
                        e.printStackTrace();
                        // 发送失败时移除客户端
                        clients.remove(client);
                    }
                }
            }
        } catch (Exception e) {
            System.err.println("分片发送消息时发生异常: " + e.getMessage());
            e.printStackTrace();
        }
    }
    
    /**
     * 发送错误信息给所有客户端
     */
    private void sendErrorToClients(String errorMessage) {
        DomSyncData errorData = new DomSyncData(
                "error", // 错误类型
                errorMessage, // 错误信息
                "", 
                "",
                getCurrentPageUrl()
        );
        String jsonData = JSON.toJSONString(errorData);
        broadcastMessage(jsonData);
    }

    // ===================== WebSocket生命周期方法 =====================
    /**
     * 客户端连接建立时触发
     */
    @Override
    public void afterConnectionEstablished(WebSocketSession session) {
        // 从会话属性中获取用户ID
        String userId = (String) session.getAttributes().get("userId");
        if (userId == null) {
            userId = "anonymous"; // 默认匿名用户
        }
        
        // 检查连接数限制
        Integer currentConnections = userConnections.getOrDefault(userId, 0);
        if (currentConnections >= MAX_CONNECTIONS_PER_USER) {
            try {
                session.close(CloseStatus.POLICY_VIOLATION.withReason("超过最大连接数限制"));
                return;
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        
        // 增加用户连接数
        userConnections.put(userId, currentConnections + 1);
        
        clients.put(session, session.getId());
        System.out.println("客户端连接成功：" + session.getId() + "，用户：" + userId);
        
        // 连接建立后不立即推送DOM，而是等待客户端发送navigate指令
        System.out.println("WebSocket连接已建立，等待客户端发送导航指令...");
    }

    /**
     * 处理客户端发送的指令（如导航、点击、输入）
     */
    @Override
    protected void handleTextMessage(WebSocketSession session, TextMessage message) {
        String payload = message.getPayload();
        
        // 记录接收到的消息
        System.out.println("收到WebSocket消息: " + payload);
        incrementCounter("websocketMessages");
        
        // 从会话属性中获取用户ID
        String userId = (String) session.getAttributes().get("userId");
        if (userId == null) {
            userId = "anonymous"; // 默认匿名用户
        }
        
        // 指令频率限制
        long currentTime = System.currentTimeMillis();
        Long lastTime = lastCommandTimes.getOrDefault(userId, 0L);
        Integer commandCount = commandCounts.getOrDefault(userId, 0);
        
        // 检查是否超过每秒最大指令数
        if (currentTime - lastTime < 1000) {
            // 同一秒内
            if (commandCount >= MAX_COMMANDS_PER_SECOND) {
                String errorMsg = "指令执行过于频繁，请稍后再试";
                System.err.println("用户 " + userId + " " + errorMsg);
                sendErrorToClients(errorMsg);
                return;
            }
            commandCounts.put(userId, commandCount + 1);
        } else {
            // 新的一秒
            lastCommandTimes.put(userId, currentTime);
            commandCounts.put(userId, 1);
        }
        
        try {
            // 检查Playwright实例是否有效
            if (!isPlaywrightInstanceValid()) {
                String errorMsg = "Playwright实例未初始化或已关闭，请刷新页面重试";
                System.err.println("用户 " + userId + " " + errorMsg);
                sendErrorToClients(errorMsg);
                return;
            }
            
            // 检查WebSocket连接状态
            if (!session.isOpen()) {
                String errorMsg = "WebSocket连接已关闭";
                System.err.println("用户 " + userId + " " + errorMsg);
                sendErrorToClients(errorMsg);
                return;
            }
            
            // 指令格式：指令类型:参数（如navigate:https://www.baidu.com）
            // 特殊处理navigate命令，避免URL中的冒号导致分割错误
            String command;
            String param;
            
            int firstColonIndex = payload.indexOf(':');
            if (firstColonIndex != -1) {
                command = payload.substring(0, firstColonIndex);
                param = payload.substring(firstColonIndex + 1);
            } else {
                command = payload;
                param = "";
            }
            
            System.out.println("处理指令: " + command + ", 参数: " + param + " (用户: " + userId + ")");
            
            switch (command) {
                case "navigate":
                    // 导航到指定URL
                    if (param == null || param.trim().isEmpty()) {
                        sendErrorToClients("导航URL不能为空");
                        return;
                    }
                    
                    // 验证URL格式
                    if (!isValidUrl(param)) {
                        sendErrorToClients("无效的URL格式: " + param);
                        return;
                    }
                    
                    try {
                        System.out.println("正在导航到URL: " + param);
                        page.navigate(param);
                        System.out.println("导航成功: " + param);
                    } catch (Exception e) {
                        String errorMsg = "导航失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "stats":
                    // 获取统计信息
                    try {
                        String stats = getStatisticsSummary();
                        DomSyncData statsData = new DomSyncData(
                                "stats", // 统计信息类型
                                stats, // 统计信息内容
                                "", 
                                "",
                                page != null ? page.url() : ""
                        );
                        String jsonData = JSON.toJSONString(statsData);
                        broadcastMessage(jsonData);
                    } catch (Exception e) {
                        String errorMsg = "获取统计信息失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "reset-stats":
                    // 重置统计信息
                    try {
                        resetAllCounters();
                        sendErrorToClients("统计信息已重置");
                    } catch (Exception e) {
                        String errorMsg = "重置统计信息失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "click":
                    // 点击指定选择器的元素（如#su）
                    try {
                        page.locator(param).click();
                    } catch (Exception e) {
                        String errorMsg = "点击元素失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "dblclick":
                    // 双击指定选择器的元素
                    try {
                        page.locator(param).dblclick();
                    } catch (Exception e) {
                        String errorMsg = "双击元素失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "hover":
                    // 悬停在指定选择器的元素上
                    try {
                        page.locator(param).hover();
                    } catch (Exception e) {
                        String errorMsg = "悬停元素失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "focus":
                    // 聚焦到指定元素
                    try {
                        page.locator(param).focus();
                    } catch (Exception e) {
                        String errorMsg = "聚焦元素失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "blur":
                    // 失去焦点
                    try {
                        page.locator(param).blur();
                    } catch (Exception e) {
                        String errorMsg = "失去焦点失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "select":
                    // 选择下拉选项（格式：选择器:值）
                    try {
                        String[] selectParts = param.split(":", 2);
                        if (selectParts.length == 2) {
                            page.locator(selectParts[0]).selectOption(selectParts[1]);
                        }
                    } catch (Exception e) {
                        String errorMsg = "选择选项失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "keydown":
                    // 键盘按下事件（格式：键码）
                    try {
                        page.keyboard().down(param);
                    } catch (Exception e) {
                        String errorMsg = "键盘按下失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "keyup":
                    // 键盘释放事件（格式：键码）
                    try {
                        page.keyboard().up(param);
                    } catch (Exception e) {
                        String errorMsg = "键盘释放失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "keypress":
                    // 键盘按键事件（格式：键码）
                    try {
                        page.keyboard().press(param);
                    } catch (Exception e) {
                        String errorMsg = "按键事件失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "scroll":
                    // 滚动页面（格式：y坐标，如500）
                    try {
                        int scrollY = Integer.parseInt(param);
                        page.evaluate("window.scrollTo(0, " + scrollY + ")");
                    } catch (Exception e) {
                        String errorMsg = "滚动页面失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "type":
                    // 输入内容（格式：选择器:内容，如#kw:Java Playwright）
                    try {
                        // 使用indexOf和substring替代split，避免内容中的冒号导致分割错误
                        int colonIndex = param.indexOf(':');
                        if (colonIndex != -1 && colonIndex < param.length() - 1) {
                            String selector = param.substring(0, colonIndex);
                            String content = param.substring(colonIndex + 1);
                            Locator inputLocator = page.locator(selector);
                            inputLocator.fill(content);
                        }
                    } catch (Exception e) {
                        String errorMsg = "输入内容失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;
                case "input":
                    // 更丰富的输入场景（格式：选择器:内容）
                    try {
                        // 使用indexOf和substring替代split，避免内容中的冒号导致分割错误
                        int colonIndex = param.indexOf(':');
                        if (colonIndex != -1 && colonIndex < param.length() - 1) {
                            String selector = param.substring(0, colonIndex);
                            String content = param.substring(colonIndex + 1);
                            Locator inputLocator = page.locator(selector);
                            inputLocator.fill(content);
                        }
                    } catch (Exception e) {
                        String errorMsg = "输入内容失败：" + e.getMessage();
                        System.err.println(errorMsg);
                        e.printStackTrace();
                        sendErrorToClients(errorMsg);
                    }
                    break;

                default:
                    sendErrorToClients("未知指令：" + command);
            }
        } catch (Exception e) {
            e.printStackTrace();
            sendErrorToClients("指令执行失败：" + e.getMessage());
        }
    }

    /**
     * 客户端连接关闭时触发
     */
    @Override
    public void afterConnectionClosed(WebSocketSession session, CloseStatus status) {
        clients.remove(session);
        System.out.println("客户端断开连接：" + session.getId());
        
        // 从会话属性中获取用户ID
        String userId = (String) session.getAttributes().get("userId");
        if (userId == null) {
            userId = "anonymous"; // 默认匿名用户
        }
        
        // 减少用户连接数
        Integer currentConnections = userConnections.getOrDefault(userId, 0);
        if (currentConnections > 0) {
            userConnections.put(userId, currentConnections - 1);
        }
    }

    /**
     * 处理WebSocket传输错误
     */
    @Override
    public void handleTransportError(WebSocketSession session, Throwable exception) {
        clients.remove(session);
        System.out.println("客户端传输错误：" + session.getId() + "，错误信息：" + exception.getMessage());
    }

    /**
     * 验证URL格式是否有效
     *
     * @param url 要验证的URL
     * @return 如果URL有效返回true，否则返回false
     */
    private boolean isValidUrl(String url) {
        if (url == null || url.trim().isEmpty()) {
            return false;
        }
        
        // 检查是否以http://或https://开头
        if (!url.toLowerCase().startsWith("http://") && !url.toLowerCase().startsWith("https://")) {
            return false;
        }
        
        // 简单的URL格式检查
        try {
            new java.net.URL(url);
            return true;
        } catch (Exception e) {
            return false;
        }
    }
    
    /**
     * 销毁资源（Spring Boot关闭时调用）
     */
    public void destroy() {
        try {
            // 关闭所有客户端连接
            for (WebSocketSession session : clients.keySet()) {
                try {
                    if (session.isOpen()) {
                        session.close(CloseStatus.NORMAL);
                    }
                } catch (Exception e) {
                    System.err.println("关闭WebSocket会话失败: " + e.getMessage());
                }
            }
            clients.clear();
        } catch (Exception e) {
            System.err.println("清理客户端连接失败: " + e.getMessage());
        }

        try {
            // 关闭Playwright资源
            if (page != null && !page.isClosed()) {
                page.close();
            }
        } catch (Exception e) {
            System.err.println("关闭页面失败: " + e.getMessage());
        }
        
        try {
            if (context != null) {
                context.close();
            }
        } catch (Exception e) {
            System.err.println("关闭浏览器上下文失败: " + e.getMessage());
        }
        
        try {
            if (browser != null && browser.isConnected()) {
                browser.close();
            }
        } catch (Exception e) {
            System.err.println("关闭浏览器失败: " + e.getMessage());
        }
        
        try {
            if (playwright != null) {
                playwright.close();
            }
        } catch (Exception e) {
            System.err.println("关闭Playwright失败: " + e.getMessage());
        }
        
        System.out.println("Playwright资源已清理完毕");
    }
    
    /**
     * 获取统计信息摘要
     * 
     * @return 包含所有统计信息的JSON字符串
     */
    public String getStatisticsSummary() {
        try {
            Map<String, Object> stats = new HashMap<>();
            stats.put("clients", clients.size());
            stats.put("userConnections", new HashMap<>(userConnections));
            stats.put("domChanges", getCounter("domChanges"));
            stats.put("websocketMessages", getCounter("websocketMessages"));
            stats.put("errors", getCounter("errors"));
            stats.put("pageLoads", getCounter("pageLoads"));
            stats.put("navigations", getCounter("navigations"));
            stats.put("timestamp", System.currentTimeMillis());
            
            if (page != null) {
                stats.put("currentPage", page.url());
            }
            
            return JSON.toJSONString(stats);
        } catch (Exception e) {
            System.err.println("获取统计信息失败: " + e.getMessage());
            e.printStackTrace();
            return "{\"error\":\"获取统计信息失败\"}";
        }
    }
    
    /**
     * 重置所有统计计数器
     */
    public void resetAllCounters() {
        try {
            String[] counters = {"domChanges", "websocketMessages", "errors", "pageLoads", "navigations"};
            for (String counter : counters) {
                resetCounter(counter);
            }
            System.out.println("所有统计计数器已重置");
        } catch (Exception e) {
            System.err.println("重置统计计数器失败: " + e.getMessage());
            e.printStackTrace();
        }
    }
    
    /**
     * 检查Playwright实例是否有效
     * 
     * @return 如果Playwright实例有效返回true，否则返回false
     */
    private boolean isPlaywrightInstanceValid() {
        try {
            return playwright != null && browser != null && page != null && !page.isClosed();
        } catch (Exception e) {
            System.err.println("检查Playwright实例状态失败: " + e.getMessage());
            return false;
        }
    }
    
    /**
     * 安全地获取当前页面URL
     * 
     * @return 当前页面URL，如果无法获取则返回空字符串
     */
    private String getCurrentPageUrl() {
        try {
            return page != null ? page.url() : "";
        } catch (Exception e) {
            System.err.println("获取当前页面URL失败: " + e.getMessage());
            return "";
        }
    }
}