package pangea.hiagent.llm;

import lombok.extern.slf4j.Slf4j;
import org.springframework.ai.chat.model.ChatModel;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import pangea.hiagent.model.LlmConfig;

import jakarta.annotation.PostConstruct;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 模型适配器管理器
 * 负责管理和选择合适的模型适配器
 */
@Slf4j
@Component
public class ModelAdapterManager {
    
    @Autowired
    private List<ModelAdapter> adapters;
    
    private Map<String, ModelAdapter> adapterMap;
    
    @PostConstruct
    public void init() {
        adapterMap = new HashMap<>();
        for (ModelAdapter adapter : adapters) {
            adapterMap.put(adapter.getProviderName().toLowerCase(), adapter);
            log.info("注册模型适配器: {}", adapter.getProviderName());
        }
    }
    
    /**
     * 根据配置创建ChatModel实例
     * 
     * @param config LLM配置
     * @return ChatModel实例
     */
    public ChatModel createChatModel(LlmConfig config) {
        log.info("开始创建ChatModel，配置信息: {}", config);
        
        if (config == null || !config.getEnabled()) {
            log.warn("无效的LLM配置或配置未启用");
            throw new IllegalArgumentException("无效的LLM配置或配置未启用");
        }
        
        String provider = config.getProvider();
        log.info("LLM提供商: {}", provider);
        
        if (provider == null || provider.isEmpty()) {
            log.warn("LLM配置缺少提供商信息");
            throw new IllegalArgumentException("LLM配置缺少提供商信息");
        }
        
        ModelAdapter adapter = adapterMap.get(provider.toLowerCase());
        if (adapter == null) {
            log.warn("不支持的LLM提供商: {}", provider);
            throw new IllegalArgumentException("不支持的LLM提供商: " + provider);
        }
        
        log.info("找到适配器: {}", adapter.getClass().getSimpleName());
        
        if (!adapter.validateConfig(config)) {
            log.warn("LLM配置验证失败: {}", provider);
            throw new IllegalArgumentException("LLM配置验证失败: " + provider);
        }
        
        try {
            log.info("调用适配器创建ChatModel");
            ChatModel model = adapter.createChatModel(config);
            log.info("成功创建ChatModel");
            return model;
        } catch (Exception e) {
            log.error("创建ChatModel失败: {}", e.getMessage(), e);
            throw new RuntimeException("创建ChatModel失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 获取指定提供商的适配器
     * 
     * @param provider 提供商名称
     * @return 模型适配器
     */
    public ModelAdapter getAdapter(String provider) {
        if (provider == null || provider.isEmpty()) {
            return null;
        }
        return adapterMap.get(provider.toLowerCase());
    }
    
    /**
     * 获取所有已注册的适配器
     * 
     * @return 适配器映射
     */
    public Map<String, ModelAdapter> getAdapters() {
        return new HashMap<>(adapterMap);
    }
}