package pangea.hiagent.tools;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;
import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;

/**
 * 订单查询工具
 * 用于查询客户订单信息
 */
@Slf4j
@Component
public class OrderQueryTool {
    
    /**
     * 根据订单号查询订单信息
     * @param orderId 订单号
     * @return 订单信息
     */
    public String queryOrderByOrderId(String orderId) {
        log.info("查询订单信息，订单号: {}", orderId);
        
        try {
            // 模拟订单查询逻辑
            // 在实际应用中，这里应该连接到订单数据库进行查询
            Map<String, Object> orderInfo = getOrderInfoFromDatabase(orderId);
            
            if (orderInfo == null) {
                return "未找到订单号为 " + orderId + " 的订单信息";
            }
            
            // 格式化订单信息
            StringBuilder result = new StringBuilder();
            result.append("订单信息查询结果：\n");
            result.append("订单号: ").append(orderInfo.get("orderId")).append("\n");
            result.append("客户姓名: ").append(orderInfo.get("customerName")).append("\n");
            result.append("商品名称: ").append(orderInfo.get("productName")).append("\n");
            result.append("订单金额: ").append(orderInfo.get("amount")).append("\n");
            result.append("下单时间: ").append(orderInfo.get("orderTime")).append("\n");
            result.append("订单状态: ").append(orderInfo.get("status")).append("\n");
            
            return result.toString();
        } catch (Exception e) {
            log.error("查询订单信息失败，订单号: {}", orderId, e);
            return "查询订单信息时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 根据客户手机号查询订单列表
     * @param phoneNumber 客户手机号
     * @return 订单列表
     */
    public String queryOrdersByPhoneNumber(String phoneNumber) {
        log.info("查询客户订单列表，手机号: {}", phoneNumber);
        
        try {
            // 模拟订单查询逻辑
            List<Map<String, Object>> orders = getOrdersFromDatabaseByPhone(phoneNumber);
            
            if (orders == null || orders.isEmpty()) {
                return "未找到手机号为 " + phoneNumber + " 的客户订单信息";
            }
            
            // 格式化订单列表
            StringBuilder result = new StringBuilder();
            result.append("客户订单列表查询结果：\n");
            
            for (int i = 0; i < orders.size(); i++) {
                Map<String, Object> order = orders.get(i);
                result.append("订单 ").append(i + 1).append(":\n");
                result.append("  订单号: ").append(order.get("orderId")).append("\n");
                result.append("  商品名称: ").append(order.get("productName")).append("\n");
                result.append("  订单金额: ").append(order.get("amount")).append("\n");
                result.append("  下单时间: ").append(order.get("orderTime")).append("\n");
                result.append("  订单状态: ").append(order.get("status")).append("\n\n");
            }
            
            return result.toString();
        } catch (Exception e) {
            log.error("查询客户订单列表失败，手机号: {}", phoneNumber, e);
            return "查询客户订单列表时发生错误: " + e.getMessage();
        }
    }
    
    /**
     * 模拟从数据库获取订单信息
     * @param orderId 订单号
     * @return 订单信息
     */
    private Map<String, Object> getOrderInfoFromDatabase(String orderId) {
        // 模拟数据库查询
        // 在实际应用中，这里应该连接真实的数据库
        
        // 模拟一些订单数据
        Map<String, Object> order1 = new HashMap<>();
        order1.put("orderId", "ORD20240101001");
        order1.put("customerName", "张三");
        order1.put("phoneNumber", "13800138001");
        order1.put("productName", "智能手表");
        order1.put("amount", "1299.00");
        order1.put("orderTime", "2024-01-01 10:30:00");
        order1.put("status", "已发货");
        
        Map<String, Object> order2 = new HashMap<>();
        order2.put("orderId", "ORD20240102002");
        order2.put("customerName", "李四");
        order2.put("phoneNumber", "13800138002");
        order2.put("productName", "无线耳机");
        order2.put("amount", "399.00");
        order2.put("orderTime", "2024-01-02 15:45:00");
        order2.put("status", "已签收");
        
        // 根据订单号返回相应订单
        if ("ORD20240101001".equals(orderId)) {
            return order1;
        } else if ("ORD20240102002".equals(orderId)) {
            return order2;
        }
        
        return null;
    }
    
    /**
     * 模拟从数据库根据手机号获取订单列表
     * @param phoneNumber 客户手机号
     * @return 订单列表
     */
    private List<Map<String, Object>> getOrdersFromDatabaseByPhone(String phoneNumber) {
        // 模拟数据库查询
        // 在实际应用中，这里应该连接真实的数据库
        
        List<Map<String, Object>> orders = new ArrayList<>();
        
        // 模拟一些订单数据
        if ("13800138001".equals(phoneNumber)) {
            Map<String, Object> order1 = new HashMap<>();
            order1.put("orderId", "ORD20240101001");
            order1.put("customerName", "张三");
            order1.put("productName", "智能手表");
            order1.put("amount", "1299.00");
            order1.put("orderTime", "2024-01-01 10:30:00");
            order1.put("status", "已发货");
            orders.add(order1);
            
            Map<String, Object> order2 = new HashMap<>();
            order2.put("orderId", "ORD20240103003");
            order2.put("customerName", "张三");
            order2.put("productName", "手机壳");
            order2.put("amount", "29.90");
            order2.put("orderTime", "2024-01-03 09:15:00");
            order2.put("status", "已付款");
            orders.add(order2);
        } else if ("13800138002".equals(phoneNumber)) {
            Map<String, Object> order = new HashMap<>();
            order.put("orderId", "ORD20240102002");
            order.put("customerName", "李四");
            order.put("productName", "无线耳机");
            order.put("amount", "399.00");
            order.put("orderTime", "2024-01-02 15:45:00");
            order.put("status", "已签收");
            orders.add(order);
        }
        
        return orders;
    }
    
    /**
     * 获取工具名称
     * @return 工具名称
     */
    public String getName() {
        return "orderQuery";
    }
    
    /**
     * 获取工具描述
     * @return 工具描述
     */
    public String getDescription() {
        return "订单查询工具，可用于查询订单信息和客户订单列表";
    }
}