package pangea.hiagent.model;

import com.baomidou.mybatisplus.annotation.TableName;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;

import java.util.List;
import java.util.HashSet;
import java.util.Set;
/**
 * Agent实体类
 * 代表一个AI智能体
 */
@Slf4j
@Data
@Builder
@NoArgsConstructor
@AllArgsConstructor
@EqualsAndHashCode(callSuper = true)
@TableName("agent")
public class Agent extends BaseEntity {    
    private static final long serialVersionUID = 1L;
    
    /**
     * Agent名称
     */
    private String name;
    
    /**
     * Agent描述
     */
    private String description;
    
    /**
     * Agent状态（active/inactive/draft）
     */
    private String status;
    
    /**
     * 默认模型
     */
    private String defaultModel;
    
    /**
     * 系统提示词
     */
    private String systemPrompt;
    
    /**
     * 提示词模板
     */
    private String promptTemplate;
    
    /**
     * 温度参数（0-2）
     */
    private Double temperature;
    
    /**
     * 最大生成Token数
     */
    private Integer maxTokens;
    
    /**
     * Top P参数
     */
    private Double topP;
    
    /**
     * Top K参数
     */
    private Integer topK;
    
    /**
     * 存在惩罚
     */
    private Double presencePenalty;
    
    /**
     * 频率惩罚
     */
    private Double frequencyPenalty;
    
    /**
     * 历史记录长度
     */
    private Integer historyLength;    
    
    /**
     * 可用工具（JSON格式）
     */
    private String tools;
    
    /**
     * 关联的RAG集合ID
     */
    private String ragCollectionId;
    
    /**
     * 是否启用RAG
     */
    private Boolean enableRag;
    
    /**
     * RAG检索的文档数量
     */
    private Integer ragTopK;
    
    /**
     * RAG相似度阈值
     */
    private Double ragScoreThreshold;
    
    /**
     * RAG提示词模板
     */
    private String ragPromptTemplate;
    
    /**
     * 是否启用ReAct Agent模式
     */
    private Boolean enableReAct;
    
    /**
     * 是否启用流式输出
     */
    private Boolean enableStreaming;
    
    /**
     * Agent所有者
     */
    private String owner;
    
    /**
     * 获取工具名称列表
     * @return 工具名称列表
     */
    public List<String> getToolNames() {
        if (tools == null || tools.isEmpty()) {
            // 返回默认工具列表而不是空列表
            return List.of("search", "calculator");
        }
        
        try {
            ObjectMapper mapper = new ObjectMapper();
            // 先尝试解析为字符串数组
            if (tools.startsWith("[") && tools.endsWith("]")) {
                List<String> toolList = mapper.readValue(tools, new TypeReference<List<String>>() {});
                // 如果解析结果为空，返回默认工具列表
                if (toolList == null || toolList.isEmpty()) {
                    return List.of("search", "calculator");
                }
                return toolList;
            } else {
                // 如果不是JSON数组格式，尝试作为单个工具名称处理
                return List.of(tools);
            }
        } catch (Exception e) {
            log.warn("解析Agent工具配置失败，Agent ID: {}, tools: {}, error: {}", getId(), tools, e.getMessage());
            // 如果解析失败，返回默认的工具列表
            return List.of("search", "calculator");
        }
    }    
    /**
     * 获取工具名称集合（去重）
     * @return 工具名称集合
     */
    public Set<String> getToolNameSet() {
        return new HashSet<>(getToolNames());
    }
}