import { defineStore } from 'pinia'
import { ref, Ref } from 'vue'
import axios, { AxiosInstance, AxiosRequestConfig, AxiosResponse, InternalAxiosRequestConfig } from 'axios'
import { useRouter } from 'vue-router'

// 定义用户信息接口
interface UserInfo {
  id?: string
  username?: string
  email?: string
  [key: string]: any
}


export const useAuthStore = defineStore('auth', () => {
  const token = ref<string | null>(localStorage.getItem('token') || null)
  const userInfo = ref<UserInfo>(JSON.parse(localStorage.getItem('userInfo') || '{}'))
  
  const api: AxiosInstance = axios.create({
    baseURL: '/api/v1'
  })
  
  api.interceptors.request.use(
    (config: InternalAxiosRequestConfig) => {
      if (token.value) {
        config.headers.Authorization = `Bearer ${token.value}`
      }
      return config
    },
    (error: any) => Promise.reject(error)
  )
  
  // 添加响应拦截器处理401错误
  api.interceptors.response.use(
    (response: AxiosResponse) => response,
    (error: any) => {
      if (error.response && error.response.status === 401) {
        // 清除认证信息
        token.value = null
        userInfo.value = {}
        localStorage.removeItem('token')
        localStorage.removeItem('userInfo')
        
        // 跳转到登录页
        if (typeof window !== 'undefined') {
          setTimeout(() => {
            window.location.href = '/login'
          }, 100)
        }
      }
      return Promise.reject(error)
    }
  )
  
  async function register(username: string, password: string, email: string): Promise<any> {
    try {
      const response: AxiosResponse = await api.post('/auth/register', { username, password, email })
      return response.data
    } catch (error: any) {
      throw new Error(error.response?.data?.message || '注册失败')
    }
  }
  
  async function login(username: string, password: string): Promise<any> {
    try {
      const response: AxiosResponse = await api.post('/auth/login', { username, password })
      const { token: newToken } = response.data.data
      token.value = newToken
      localStorage.setItem('token', newToken)
      return response.data
    } catch (error: any) {
      throw new Error(error.response?.data?.message || '登录失败')
    }
  }
  
  /**
   * OAuth2 授权流程
   * 重定向用户到授权服务器
   */
  async function loginWithOAuth2(providerName: string): Promise<void> {
    try {
      // 调用后端端点得到授权 URL
      const response: AxiosResponse = await api.get(`/auth/oauth2/authorize?providerName=${providerName}`)
      // 后端分路归结会停止进一步的处理，此处是正常的
      // 实际上此调用为无效的。前端应该直接渐进到授权端点
    } catch (error: any) {
      throw new Error(error.response?.data?.message || 'OAuth2 授权失败')
    }
  }
  
  /**
   * OAuth2 回调函数
   * 处理来自授权服务器的回调
   */
  function handleOAuth2Callback(): Promise<any> {
    return new Promise((resolve, reject) => {
      // 从 URL 参数中提取授权码
      const params = new URLSearchParams(window.location.search)
      const authCode = params.get('code')
      const providerName = params.get('provider')
      const error = params.get('error')
      
      if (error) {
        reject(new Error(`OAuth2 错误: ${error}`))
        return
      }
      
      if (!authCode || !providerName) {
        reject(new Error('不完整的 OAuth2 回调参数'))
        return
      }
      
      // 使用授权码进行令牌交换
      loginWithOAuth2Code(authCode, providerName)
        .then(resolve)
        .catch(reject)
    })
  }
  
  /**
   * 使用授权码不什止迟不何敷不洘华丞身式末身
   */
  async function loginWithOAuth2Code(authCode: string, providerName: string): Promise<any> {
    try {
      const response: AxiosResponse = await api.post('/auth/oauth2/token', {
        authorizationCode: authCode,
        providerName: providerName
      })
      
      const { token: newToken } = response.data.data
      token.value = newToken
      localStorage.setItem('token', newToken)
      
      return response.data
    } catch (error: any) {
      throw new Error(error.response?.data?.message || 'OAuth2 认证失败')
    }
  }
  
  function logout(): void {
    token.value = null
    userInfo.value = {}
    localStorage.removeItem('token')
    localStorage.removeItem('userInfo')
  }
  
  function setUserInfo(info: UserInfo): void {
    userInfo.value = info
    localStorage.setItem('userInfo', JSON.stringify(info))
  }
  
  // 添加便捷方法
  const get = (url: string, config: AxiosRequestConfig = {}): Promise<AxiosResponse> => api({ method: 'get', url, ...config })
  const post = (url: string, data?: any, config: AxiosRequestConfig = {}): Promise<AxiosResponse> => api({ method: 'post', url, data, ...config })
  const put = (url: string, data?: any, config: AxiosRequestConfig = {}): Promise<AxiosResponse> => api({ method: 'put', url, data, ...config })
  const del = (url: string, config: AxiosRequestConfig = {}): Promise<AxiosResponse> => api({ method: 'delete', url, ...config })
  
  return {
    token,
    userInfo,
    register,
    login,
    loginWithOAuth2,
    handleOAuth2Callback,
    loginWithOAuth2Code,
    logout,
    setUserInfo,
    api,
    get,
    post,
    put,
    del
  }
})