// DOM工具函数

/**
 * DOM工具函数模块
 * 提供DOM操作相关的实用函数
 */

/**
 * 获取元素的选择器（根据后端协议重构）
 * @param element - 要获取选择器的DOM元素
 * @returns 元素的CSS选择器字符串
 */
export const getElementSelector = (element: Element): string => {
  if (!element || element === document.documentElement || element === document.body) {
    return 'body';
  }
  
  // 优先使用ID选择器
  if (element.id) {
    try {
      return '#' + CSS.escape(element.id);
    } catch (e) {
      // 如果CSS.escape失败，使用简单的ID选择器
      return '#' + element.id;
    }
  }
  
  // 使用类名选择器
  if (element.className && typeof element.className === 'string') {
    const classes = element.className.split(' ').filter(cls => cls.length > 0);
    if (classes.length > 0) {
      try {
        return '.' + classes.map(cls => CSS.escape(cls)).join('.');
      } catch (e) {
        // 如果CSS.escape失败，使用简单的类名选择器
        return '.' + classes.join('.');
      }
    }
  }
  
  // 使用标签名选择器，并结合nth-child定位
  let selector = element.tagName.toLowerCase();
  const parent = element.parentElement;
  
  if (parent) {
    // 获取同级同标签元素的数量
    const siblings = Array.from(parent.children).filter(child => 
      child.tagName === element.tagName);
    
    if (siblings.length > 1) {
      // 计算当前元素在同级元素中的位置（1-based）
      const index = Array.from(siblings).indexOf(element) + 1;
      selector += `:nth-child(${index})`;
    }
  }
  
  // 如果有父元素，递归构建更精确的选择器
  if (parent && parent !== document.body) {
    const parentSelector = getElementSelector(parent);
    if (parentSelector) {
      selector = `${parentSelector} > ${selector}`;
    }
  }
  
  return selector;
};

/**
 * 检查脚本是否安全执行
 * @param scriptText - 要检查的脚本文本
 * @returns 如果脚本安全则返回true，否则返回false
 */
export const isScriptSafe = (scriptText: string): boolean => {
  // 简单的安全检查，防止明显的恶意脚本执行
  const unsafePatterns = [
    /document\.cookie/i,
    /localStorage/i,
    /sessionStorage/i,
    /indexedDB/i,
    /navigator\.sendBeacon/i,
    /XMLHttpRequest/i,
    /fetch\s*\(/i,
    /eval\s*\(/i,
    /Function\s*\(/i,
    /setTimeout\s*\([^,]+,[^,]+\)/i,
    /setInterval\s*\([^,]+,[^,]+\)/i,
    /location\s*=\s*/i,
    /window\.open/i,
    /document\.write/i
  ];
  
  // 检查是否存在不安全模式
  return !unsafePatterns.some(pattern => pattern.test(scriptText));
};

/**
 * 验证URL格式是否有效
 * @param urlString - 要验证的URL字符串
 * @returns 如果URL有效则返回true，否则返回false
 */
export const isValidUrl = (urlString: string): boolean => {
  if (!urlString || urlString.trim() === '') {
    return false;
  }
  
  let url = urlString.trim();
  
  // 检查是否以http://或https://开头
  if (!url.toLowerCase().startsWith('http://') && !url.toLowerCase().startsWith('https://')) {
    // 如果没有协议前缀，自动添加https://
    url = 'https://' + url;
  }
      
  // 确保URL末尾没有多余的斜杠（除了域名根路径）
  if (url.endsWith('/') && url.length > 8) { // 8是'http://a'的长度
    url = url.slice(0, -1);
  }
  
  // 使用URL构造函数验证URL格式
  try {
    new URL(url);
    return true;
  } catch (e) {
    return false;
  }
};