// 统一事件处理器
import type { TimelineEvent } from '../types/timeline'
import { EventProcessingService } from './EventProcessingService'
import { EventDeduplicationService } from './EventDeduplicationService'
import { OptimizedEventProcessingService } from './OptimizedEventProcessingService'

export class UnifiedEventProcessor {
  private eventProcessingService: EventProcessingService
  private eventDeduplicationService: EventDeduplicationService
  private optimizedEventProcessingService: OptimizedEventProcessingService
  private eventHandlers: Array<(event: TimelineEvent) => void> = []
  private processedEvents: TimelineEvent[] = []

  constructor() {
    this.eventProcessingService = new EventProcessingService()
    this.eventDeduplicationService = new EventDeduplicationService()
    this.optimizedEventProcessingService = new OptimizedEventProcessingService()
  }

  /**
   * 处理接收到的原始事件数据
   * @param rawData 原始事件数据
   * @returns 处理后的标准化事件对象
   */
  processRawEvent(rawData: any): TimelineEvent | null {
    try {
      // 验证数据
      if (!rawData || typeof rawData !== 'object') {
        console.warn('[UnifiedEventProcessor] 无效的事件数据:', rawData)
        return null
      }

      // 检查是否为重复事件
      if (this.eventDeduplicationService.isDuplicateEvent(rawData)) {
        console.log('[UnifiedEventProcessor] 跳过重复事件:', rawData.type, rawData.title)
        return null
      }

      // 处理事件类型转换
      const processedEvent = this.eventProcessingService.processEventType(rawData)

      // 标准化事件对象（使用优化的服务）
      const normalizedEvent = this.optimizedEventProcessingService.normalizeEvent(processedEvent)

      // 添加到已处理事件列表
      this.processedEvents.push(normalizedEvent)

      // 限制已处理事件列表大小以避免内存泄漏
      if (this.processedEvents.length > 1000) {
        this.processedEvents.shift()
      }

      return normalizedEvent
    } catch (error) {
      console.error('[UnifiedEventProcessor] 处理事件数据时发生错误:', error, '原始数据:', rawData)
      return null
    }
  }

  /**
   * 注册事件处理器
   * @param handler 事件处理器函数
   */
  registerHandler(handler: (event: TimelineEvent) => void): void {
    this.eventHandlers.push(handler)
  }

  /**
   * 分发事件给所有注册的处理器
   * @param event 事件对象
   */
  dispatchEvent(event: TimelineEvent): void {
    this.eventHandlers.forEach(handler => {
      try {
        handler(event)
      } catch (error) {
        console.error('[UnifiedEventProcessor] 事件处理器执行错误:', error)
      }
    })
  }

  /**
   * 处理并分发事件
   * @param rawData 原始事件数据
   */
  processAndDispatch(rawData: any): void {
    const event = this.processRawEvent(rawData)
    if (event) {
      this.dispatchEvent(event)
    }
  }

  /**
   * 清除已处理事件列表
   */
  clearProcessedEvents(): void {
    this.processedEvents = []
    this.eventDeduplicationService.clearEventHashSet()
  }

  /**
   * 获取性能统计信息
   * @returns 性能统计信息
   */
  getPerformanceStats(): { totalProcessed: number; totalReused: number; reuseRate: number } {
    return this.optimizedEventProcessingService.getPerformanceStats();
  }
}