// 优化的事件处理服务
import { ObjectPoolService } from './ObjectPoolService';
import type { TimelineEvent } from '../types/timeline';

export class OptimizedEventProcessingService {
  private eventObjectPool: ObjectPoolService<Record<string, any>>;
  private MAX_POOL_SIZE: number = 100;
  private totalEventsProcessed: number = 0;
  private totalEventsReused: number = 0;

  constructor() {
    // 创建对象池用于事件对象
    this.eventObjectPool = new ObjectPoolService<Record<string, any>>(
      () => ({}), // 工厂函数创建空对象
      (obj) => {  // 重置函数清空对象属性
        for (const key in obj) {
          if (obj.hasOwnProperty(key)) {
            delete obj[key];
          }
        }
      },
      this.MAX_POOL_SIZE
    );
  }

  /**
   * 标准化事件对象（使用对象池优化）
   * @param event 原始事件数据
   * @returns 标准化的事件对象
   */
  normalizeEvent(event: any): TimelineEvent {
    // 从对象池获取对象
    const normalizedEvent = this.eventObjectPool.acquire();
    
    try {
      // 确保时间戳存在
      const timestamp = event.timestamp || Date.now();

      // 根据事件类型创建相应类型的事件对象
      switch (event.type) {
        case 'thought':
          Object.assign(normalizedEvent, {
            type: 'thought',
            title: event.title || '思考事件',
            content: event.content || '',
            thinkingType: event.thinkingType,
            metadata: event.metadata,
            timestamp
          });
          break;

        case 'tool_call':
          Object.assign(normalizedEvent, {
            type: 'tool_call',
            title: event.title || '工具调用',
            toolName: event.toolName || '',
            toolAction: event.toolAction || '',
            toolInput: event.toolInput,
            toolStatus: event.toolStatus,
            metadata: event.metadata,
            timestamp
          });
          break;

        case 'tool_result':
          Object.assign(normalizedEvent, {
            type: 'tool_result',
            title: event.title || '工具结果',
            toolName: event.toolName || '',
            toolAction: event.toolAction || '',
            toolOutput: event.toolOutput,
            toolStatus: event.toolStatus,
            executionTime: event.executionTime,
            metadata: event.metadata,
            timestamp
          });
          break;

        case 'tool_error':
          Object.assign(normalizedEvent, {
            type: 'tool_error',
            title: event.title || '工具错误',
            toolName: event.toolName || '',
            errorMessage: event.errorMessage || '',
            errorCode: event.errorCode,
            metadata: event.metadata,
            timestamp
          });
          break;

        case 'embed':
          Object.assign(normalizedEvent, {
            type: 'embed',
            title: event.title || '嵌入内容',
            embedUrl: event.embedUrl || '',
            embedType: event.embedType,
            embedTitle: event.embedTitle,
            embedHtmlContent: event.embedHtmlContent,
            metadata: event.metadata,
            timestamp
          });
          break;

        default:
          Object.assign(normalizedEvent, {
            type: event.type || 'thought',
            title: event.title || '未命名事件',
            metadata: event.metadata,
            timestamp
          });
          break;
      }

      this.totalEventsProcessed++;
      return normalizedEvent as TimelineEvent;
    } catch (error) {
      // 如果出现错误，将对象归还到池中
      this.eventObjectPool.release(normalizedEvent);
      throw error;
    }
  }

  /**
   * 处理完事件后，将对象归还到池中
   * @param obj 事件对象
   */
  releaseEventObject(obj: Record<string, any>): void {
    if (obj && typeof obj === 'object') {
      // 清空对象属性
      for (const key in obj) {
        if (obj.hasOwnProperty(key)) {
          delete obj[key];
        }
      }
      
      // 如果对象池未满，将对象放回池中
      if (this.eventObjectPool.size() < this.MAX_POOL_SIZE) {
        this.eventObjectPool.release(obj);
        this.totalEventsReused++;
      }
    }
  }

  /**
   * 获取性能统计信息
   * @returns 性能统计信息
   */
  getPerformanceStats(): { totalProcessed: number; totalReused: number; reuseRate: number } {
    const reuseRate = this.totalEventsProcessed > 0 
      ? (this.totalEventsReused / this.totalEventsProcessed) * 100 
      : 0;
      
    return {
      totalProcessed: this.totalEventsProcessed,
      totalReused: this.totalEventsReused,
      reuseRate: parseFloat(reuseRate.toFixed(2))
    };
  }

  /**
   * 清空统计信息
   */
  clearStats(): void {
    this.totalEventsProcessed = 0;
    this.totalEventsReused = 0;
  }
}