package pangea.hiagent.utils;

import lombok.extern.slf4j.Slf4j;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Component;

import java.io.Serializable;

/**
 * 用户上下文传播工具类
 * 用于在异步线程间传播用户认证信息
 */
@Slf4j
@Component
public class UserContextPropagationUtil {

    /**
     * 用户上下文持有者类，用于在异步线程间传递认证信息
     */
    public static class UserContextHolder implements Serializable {
        private static final long serialVersionUID = 1L;
        
        private final Authentication authentication;
        
        public UserContextHolder(Authentication authentication) {
            this.authentication = authentication;
        }
        
        public Authentication getAuthentication() {
            return authentication;
        }
    }
    
    /**
     * 捕获当前线程的用户上下文
     * @return 用户上下文持有者对象
     */
    public static UserContextHolder captureUserContext() {
        try {
            Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
            if (authentication != null) {
                log.debug("捕获到当前线程的用户认证信息: {}", authentication.getPrincipal());
                return new UserContextHolder(authentication);
            } else {
                log.debug("当前线程无用户认证信息");
                return null;
            }
        } catch (Exception e) {
            log.error("捕获用户上下文时发生异常", e);
            return null;
        }
    }
    
    /**
     * 将用户上下文传播到当前线程
     * @param userContextHolder 用户上下文持有者对象
     */
    public static void propagateUserContext(UserContextHolder userContextHolder) {
        try {
            if (userContextHolder != null) {
                Authentication authentication = userContextHolder.getAuthentication();
                if (authentication != null) {
                    SecurityContext context = SecurityContextHolder.createEmptyContext();
                    context.setAuthentication(authentication);
                    SecurityContextHolder.setContext(context);
                    log.debug("已将用户认证信息传播到当前线程: {}", authentication.getPrincipal());
                } else {
                    log.debug("用户上下文持有者中的认证信息为空");
                }
            } else {
                log.debug("用户上下文持有者为空");
            }
        } catch (Exception e) {
            log.error("传播用户上下文时发生异常", e);
        }
    }
    
    /**
     * 清理当前线程的用户上下文
     */
    public static void clearUserContext() {
        try {
            SecurityContextHolder.clearContext();
            log.debug("已清理当前线程的用户上下文");
        } catch (Exception e) {
            log.error("清理用户上下文时发生异常", e);
        }
    }
}