package pangea.hiagent.web.controller;

import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.agent.service.AgentChatService;
import pangea.hiagent.agent.service.AgentValidationService;
import pangea.hiagent.common.utils.UserUtils;
import pangea.hiagent.model.Agent;
import pangea.hiagent.web.dto.ChatRequest;
import pangea.hiagent.web.service.AgentService;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;

/**
 * Agent 对话控制器
 * 职责：接收HTTP请求并转发给AgentChatService处理
 */
@Slf4j
@RestController
@RequestMapping("/api/v1/agent")
public class AgentChatController {
    
    private final AgentChatService agentChatService;
    private final AgentService agentService;
    
    public AgentChatController(AgentChatService agentChatService, AgentService agentService) {
        this.agentChatService = agentChatService;
        this.agentService = agentService;
    }
    
    /**
     * 流式对话接口
     * 
     * @param agentId Agent ID
     * @param chatRequest 对话请求
     * @param response HTTP响应
     * @return SSE emitter
     */
    @PostMapping("/chat-stream")
    public SseEmitter chatStream(
            @RequestParam @NotBlank(message = "Agent ID不能为空") String agentId,
            @RequestBody @Valid ChatRequest chatRequest,
            HttpServletResponse response) {
        log.info("接收到流式对话请求，AgentId: {}", agentId);
        
        // 在主线程中完成权限检查，避免在异步线程中触发Spring Security异常
        String userId = UserUtils.getCurrentUserId();
        if (userId == null) {
            log.warn("用户未认证，无法执行Agent对话");
            throw new org.springframework.security.access.AccessDeniedException("用户未认证");
        }
        
        // 验证Agent存在性和权限
        Agent agent = agentService.getAgent(agentId);
        if (agent == null) {
            log.warn("Agent不存在: {}", agentId);
            throw new IllegalArgumentException("Agent不存在");
        }
        
        // 检查权限
        if (!agent.getOwner().equals(userId) && !UserUtils.isAdminUser(userId)) {
            log.warn("用户 {} 无权限访问Agent: {}", userId, agentId);
            throw new org.springframework.security.access.AccessDeniedException("无权限访问该Agent");
        }
        
        // 权限验证通过，调用异步处理
        return agentChatService.handleChatStream(agentId, chatRequest, response);
    }
}