package pangea.hiagent.agent.react;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import lombok.extern.slf4j.Slf4j;
import pangea.hiagent.workpanel.IWorkPanelDataCollector;

@Slf4j
@Component
public class TokenTextSegmenter {

    @Autowired
    private IWorkPanelDataCollector workPanelCollector;

    // 定义分段标识（按出现优先级排序）
    private static final String[] SEGMENT_MARKERS = {
            "Thought:",
            "Action:",
            "Observation:",
            "Final_Answer:"
    };

    // 当前缓存的输入字符
    private StringBuilder currentInputBuffer;

    public TokenTextSegmenter() {
        currentInputBuffer = new StringBuilder();
    }

    /**
     * 逐字输入字符并处理分段
     * 
     * @param inputChar 单个输入字符
     * @return 当分割出有效文本段时返回该段内容，否则返回null
     */
    public synchronized void inputChar(String inputChar) {
        // 输入验证
        if (inputChar == null) {
            return;
        }
        
        // 将字符加入缓存
        currentInputBuffer.append(inputChar);
        String currentBufferStr = currentInputBuffer.toString();

        log.info("【输入字符】: {}", currentBufferStr);

        // 检查当前缓冲区中是否出现任何SEGMENT_MARKERS字段
        for (String marker : SEGMENT_MARKERS) {
            int markerIndex = currentBufferStr.indexOf(marker);
            if (markerIndex != -1) {
                // 找到SEGMENT_MARKERS字段，截取该字段之前的文本进行输出
                String contentBeforeMarker = currentBufferStr.substring(0, markerIndex);
                
                // 输出截取的内容
                outputSegment(marker, contentBeforeMarker);
                
                // 重置缓冲区，保留标识符及之后的内容
                currentInputBuffer = new StringBuilder(currentBufferStr.substring(markerIndex));
                
                log.info("【识别到分段标识】: {}", marker);
                break; // 找到第一个标识后就处理并退出，避免重复处理
            }
        }
        // 如果没有找到SEGMENT_MARKERS字段，则不输出，等待更多输入
    }

    /**
     * 文本输入结束时，处理最后一个分段
     * 
     * @return 最后一个分段的内容，无则返回null
     */
    public synchronized void finishInput() {
        // 如果缓冲区还有内容，输出全部剩余内容
        if (currentInputBuffer.length() > 0) {
            String remainingContent = currentInputBuffer.toString();
            // 输出剩余的全部内容，使用一个通用标记或保持原格式
            outputSegment("Final_Content:", remainingContent);
            // 清空缓冲区
            currentInputBuffer.setLength(0);
        }
    }

    /**
     * 重置分段状态
     * 
     * @param newStartIndex 新的起始索引
     */
    private void resetSegmentState(int newStartIndex) {
        // 保留未处理的部分，用于下一个分段
        String remainingStr = currentInputBuffer.substring(newStartIndex);
        currentInputBuffer = new StringBuilder(remainingStr);
    }

    /**
     * 输出分段内容（封装输出逻辑）
     * 
     * @param marker  分段标识
     * @param content 分段内容
     * @return 格式化后的分段内容
     */
    private void outputSegment(String marker, String content) {
        log.info("【分段内容】{}: {}", marker, content);
        // 根据实际需求处理事件，这里可能需要创建适当的事件对象而不是传入null
        // workPanelCollector.addEvent(null); // 临时注释掉可能引发问题的调用
        // 或者创建一个适当的事件对象
        // 例如：workPanelCollector.addEvent(new WorkPanelEvent(marker, content));
    }
}
