package pangea.hiagent.workpanel.event;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import pangea.hiagent.web.dto.EmbedEvent;
import pangea.hiagent.web.dto.LogEvent;
import pangea.hiagent.web.dto.ResultEvent;
import pangea.hiagent.web.dto.ThoughtEvent;
import pangea.hiagent.web.dto.ToolEvent;
import pangea.hiagent.web.dto.WorkPanelEvent;
import pangea.hiagent.workpanel.WorkPanelUtils;
import pangea.hiagent.workpanel.data.ErrorEventDataBuilder;
import pangea.hiagent.workpanel.data.MapPoolService;
import pangea.hiagent.workpanel.data.TokenEventDataBuilder;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.io.IOException;

/**
 * 统一事件服务
 * 整合事件创建、构建、发送等功能，事件处理架构
 */
@Slf4j
@Service
public class EventService {
    
    @Autowired
    private MapPoolService mapPoolService;
    
    @Autowired
    private ErrorEventDataBuilder errorEventDataBuilder;
    
    @Autowired
    private TokenEventDataBuilder tokenEventDataBuilder;
    
    /**
     * 工具调用状态跟踪映射
     * key: toolName + timestamp
     * value: pending状态的事件
     */
    private final Map<String, ToolEvent> pendingToolCalls = new ConcurrentHashMap<>();
    
    // 日期格式化对象，用于日志输出
    private static final java.text.SimpleDateFormat DATE_FORMAT = new java.text.SimpleDateFormat("HH:mm:ss");
    
    /**
     * 记录工具调用开始事件
     * 
     * @param toolName 工具名称
     * @param input 工具输入参数
     * @return 工具调用开始事件
     */
    public ToolEvent recordToolCallStart(String toolName, Object input) {
        try {
            long currentTime = System.currentTimeMillis();
            ToolEvent event = ToolEvent.builder()
                    .type("tool_call")
                    .timestamp(currentTime)
                    .toolName(toolName != null ? toolName : "未知工具")
                    .toolInput(WorkPanelUtils.convertToMap(input))
                    .toolStatus("pending")
                    .build();
            
            // 跟踪pending状态的工具调用
            String key = generateToolCallKey(toolName, currentTime);
            pendingToolCalls.put(key, event);
            
            // 输出工具调用日志
            logToolCallStart(toolName, input, currentTime);
            
            log.debug("已记录工具调用开始: 工具={}, 状态=pending", toolName);
            return event;
        } catch (Exception e) {
            log.error("记录工具调用开始时发生错误: toolName={}", toolName, e);
            return null;
        }
    }
    
    /**
     * 记录工具调用完成事件
     * 
     * @param toolName 工具名称
     * @param output 工具输出结果
     * @param status 执行状态（success/failure）
     * @param executionTime 执行时间（毫秒）
     * @return 工具调用完成事件
     */
    public ToolEvent recordToolCallComplete(String toolName, Object output, String status, Long executionTime) {
        return handleToolCallCompletion(toolName, output, status, executionTime);
    }
    
    /**
     * 记录工具调用错误事件
     * 
     * @param toolName 工具名称
     * @param errorMessage 错误信息
     * @return 工具调用错误事件
     */
    public ToolEvent recordToolCallError(String toolName, String errorMessage) {
        return handleToolCallError(toolName, errorMessage);
    }
    
    /**
     * 记录思考事件
     * 
     * @param content 思考内容
     * @param thinkingType 思考类型（分析、规划、执行等）
     * @return 思考事件
     */
    public ThoughtEvent recordThinking(String content, String thinkingType) {
        try {
            // 过滤掉空内容，避免记录过多无关信息
            if (content == null || content.trim().isEmpty()) {
                return null;
            }
            
            ThoughtEvent event = ThoughtEvent.builder()
                    .type("thought")
                    .timestamp(System.currentTimeMillis())
                    .content(content)
                    .thinkingType(thinkingType != null ? thinkingType : "reasoning")
                    .build();
            
            log.debug("已记录思考过程: 类型={}, 内容={}", thinkingType, content);
            return event;
        } catch (Exception e) {
            log.error("记录思考过程时发生错误: content={}, type={}", content, thinkingType, e);
            return null;
        }
    }
    
    /**
     * 记录最终答案事件
     * 
     * @param content 最终答案内容
     * @return 最终答案事件
     */
    public ResultEvent recordFinalAnswer(String content) {
        try {
            ResultEvent event = ResultEvent.builder()
                    .type("result")
                    .timestamp(System.currentTimeMillis())
                    .content(content)
                    .build();
            
            log.debug("已记录最终答案: 内容={}", content);
            return event;
        } catch (Exception e) {
            log.error("记录最终答案时发生错误: content={}", content, e);
            return null;
        }
    }
    
    /**
     * 记录日志事件
     * 
     * @param message 日志消息
     * @param level 日志级别（info/warn/error/debug）
     * @return 日志事件
     */
    public LogEvent recordLog(String message, String level) {
        try {
            LogEvent event = LogEvent.builder()
                    .type("log")
                    .timestamp(System.currentTimeMillis())
                    .content(message)
                    .logLevel(level != null ? level : "info")
                    .build();
            
            log.debug("已记录日志: 级别={}, 消息={}", level, message);
            return event;
        } catch (Exception e) {
            log.error("记录日志时发生错误: message={}, level={}", message, level, e);
            return null;
        }
    }
    
    /**
     * 记录嵌入事件
     * 
     * @param url 嵌入资源URL（可选）
     * @param type MIME类型
     * @param title 嵌入标题
     * @param htmlContent HTML内容（可选）
     * @return 嵌入事件
     */
    public EmbedEvent recordEmbed(String url, String type, String title, String htmlContent) {
        try {
            EmbedEvent event = EmbedEvent.builder()
                    .type("embed")
                    .timestamp(System.currentTimeMillis())
                    .embedUrl(url)
                    .embedType(type)
                    .embedTitle(title != null ? title : "网页预览")
                    .embedHtmlContent(htmlContent)
                    .build();
            
            log.debug("已记录嵌入事件: 标题={}, URL={}", title, url);
            return event;
        } catch (Exception e) {
            log.error("记录嵌入事件时发生错误: title={}, url={}", title, url, e);
            return null;
        }
    }
    
    /**
     * 构建工作面板事件数据
     * 
     * @param event 工作面板事件
     * @return 事件数据
     */
    public Map<String, Object> buildWorkPanelEventData(WorkPanelEvent event) {
        if (event == null) {
            return null;
        }
        
        // 从对象池获取Map，避免频繁创建对象
        Map<String, Object> data = acquireMap();
        
        // 设置基础属性
        data.put("type", event.getType());
        data.put("eventType", event.getType());
        data.put("timestamp", event.getTimestamp());
        
        // 根据事件类型设置特有属性
        setEventSpecificProperties(event, data);
        
        // 处理事件类型特定逻辑
        processEventTypeSpecifics(event, data);
        
        return data;
    }
    
    /**
     * 发送工作面板事件给指定的SSE连接
     * 
     * @param emitter SSE发射器
     * @param event 工作面板事件
     * @throws IOException IO异常
     */
    public void sendWorkPanelEvent(SseEmitter emitter, WorkPanelEvent event) throws IOException {
        if (event == null) {
            log.warn("工作面板事件为空，无法发送事件");
            return;
        }

        try {
            // 构建事件数据
            Map<String, Object> data = buildWorkPanelEventData(event);
            
            if (data != null) {
                log.debug("准备发送工作面板事件: 类型={}, 事件内容={}", event.getType(), event);
                log.debug("事件数据: {}", data);
                
                // 发送事件
                emitter.send(org.springframework.web.servlet.mvc.method.annotation.SseEmitter.event().name("message").data(data));
                
                log.debug("工作面板事件发送成功: 类型={}", event.getType());
            } else {
                log.warn("构建事件数据失败，无法发送事件: 类型={}", event.getType());
            }
        } catch (Exception e) {
            // 记录详细错误信息，但不中断主流程
            log.error("发送工作面板事件失败: 类型={}, 错误={}", event.getType(), e.getMessage(), e);
            
            // 如果是连接已关闭的异常，重新抛出以便上层处理
            if (e instanceof IllegalStateException && e.getMessage() != null && 
                e.getMessage().contains("Emitter is already completed")) {
                throw e;
            }
        }
    }
    
    /**
     * 发送工作面板事件给指定用户
     * 
     * @param emitter SSE发射器
     * @param event 工作面板事件
     */
    public void sendWorkPanelEventToUser(SseEmitter emitter, WorkPanelEvent event) {
        log.debug("开始发送工作面板事件");
        
        if (emitter != null) {
            try {
                // 直接向当前 emitter 发送事件
                sendWorkPanelEvent(emitter, event);
                log.debug("已发送工作面板事件到客户端: {}", event.getType());
            } catch (IOException e) {
                log.error("发送工作面板事件失败: {}", e.getMessage(), e);
            }
        } else {
            log.debug("连接已失效，跳过发送事件: {}", event.getType());
        }
    }
    
    /**
     * 发送错误事件
     * 
     * @param emitter SSE发射器
     * @param errorMessage 错误信息
     * @throws IOException IO异常
     */
    public void sendErrorEvent(SseEmitter emitter, String errorMessage) throws IOException {
        if (emitter == null) {
            log.warn("SSE发射器为空，无法发送错误事件");
            return;
        }
        
        try {
            // 构建错误事件数据
            Map<String, Object> data = errorEventDataBuilder.createErrorEventData(errorMessage);
            
            if (data != null) {
                log.debug("准备发送错误事件: 错误信息={}", errorMessage);
                log.debug("错误事件数据: {}", data);
                
                // 发送事件
                emitter.send(SseEmitter.event().name("error").data(data));
                
                log.debug("错误事件发送成功");
            } else {
                log.warn("构建错误事件数据失败，无法发送事件");
            }
        } catch (Exception e) {
            // 记录详细错误信息，但不中断主流程
            log.error("发送错误事件失败: 错误信息={}, 错误={}", errorMessage, e.getMessage(), e);
            
            // 如果是连接已关闭的异常，重新抛出以便上层处理
            if (e instanceof IllegalStateException && e.getMessage() != null && 
                e.getMessage().contains("Emitter is already completed")) {
                throw e;
            }
        }
    }
    
    /**
     * 发送Token事件
     * 
     * @param emitter SSE发射器
     * @param token Token内容
     * @throws IOException IO异常
     */
    public void sendTokenEvent(SseEmitter emitter, String token) throws IOException {
        if (emitter == null) {
            log.warn("SSE发射器为空，无法发送token事件");
            return;
        }
        
        try {
            // 构建token事件数据
            Map<String, Object> data = tokenEventDataBuilder.createOptimizedTokenEventData(token);
            
            if (data != null) {
                // log.debug("准备发送token事件: token长度={}", token != null ? token.length() : 0);
                
                // 发送事件
                emitter.send(SseEmitter.event().name("token").data(data));
                
                // log.debug("token事件发送成功");
            } else {
                log.warn("构建token事件数据失败，无法发送事件");
            }
        } catch (Exception e) {
            // 记录详细错误信息，但不中断主流程
            log.error("发送token事件失败: token长度={}, 错误={}", token != null ? token.length() : 0, e.getMessage(), e);
            
            // 如果是连接已关闭的异常，重新抛出以便上层处理
            if (e instanceof IllegalStateException && e.getMessage() != null && 
                e.getMessage().contains("Emitter is already completed")) {
                throw e;
            }
        }
    }
    
    // ==================== 私有方法 ====================
    
    /**
     * 处理工具调用完成
     * 
     * @param toolName 工具名称
     * @param output 工具输出结果
     * @param status 执行状态
     * @param executionTime 执行时间
     * @return 工作面板事件
     */
    private ToolEvent handleToolCallCompletion(String toolName, Object output, String status, Long executionTime) {
        long currentTime = System.currentTimeMillis();
        
        // 查找对应的pending事件
        ToolEvent pendingEvent = findPendingToolCall(toolName);
        if (pendingEvent != null) {
            // 移除跟踪记录
            String key = generateToolCallKey(toolName, pendingEvent.getTimestamp());
            pendingToolCalls.remove(key);
            
            // 更新现有事件
            updatePendingEventForCompletion(pendingEvent, output, status, executionTime, currentTime);
            
            // 输出工具调用完成日志
            logToolCallCompletion(toolName, output, status, executionTime, currentTime);
            
            log.debug("已记录工具调用完成: 工具={}, 状态={}, 执行时间={}ms", toolName, status, executionTime);
            return pendingEvent;
        } else {
            // 如果没有对应的pending事件，创建一个新事件
            ToolEvent event = createNewCompletionEvent(toolName, output, status, executionTime, currentTime);
            
            log.debug("已记录工具调用完成（无对应pending事件）: 工具={}, 状态={}, 执行时间={}ms", toolName, status, executionTime);
            return event;
        }
    }
    
    /**
     * 处理工具调用错误
     * 
     * @param toolName 工具名称
     * @param errorMessage 错误信息
     * @return 工作面板事件
     */
    private ToolEvent handleToolCallError(String toolName, String errorMessage) {
        // 查找对应的pending事件
        ToolEvent pendingEvent = findPendingToolCall(toolName);
        if (pendingEvent != null) {
            // 移除跟踪记录
            String key = generateToolCallKey(toolName, pendingEvent.getTimestamp());
            pendingToolCalls.remove(key);
            
            // 更新现有事件为错误状态
            pendingEvent.setType("tool_error");
            // 使用反射设置content字段
            try {
                java.lang.reflect.Field contentField = WorkPanelEvent.class.getDeclaredField("content");
                contentField.setAccessible(true);
                contentField.set(pendingEvent, errorMessage);
            } catch (Exception e) {
                // 如果反射失败，忽略错误
            }
            pendingEvent.setToolStatus("failure");
            pendingEvent.setTimestamp(System.currentTimeMillis());
            
            log.debug("已记录工具调用错误: 工具={}, 错误={}", toolName, errorMessage);
            return pendingEvent;
        } else {
            // 如果没有对应的pending事件，创建一个新事件
            ToolEvent event = ToolEvent.builder()
                    .type("tool_error")
                    .timestamp(System.currentTimeMillis())
                    .toolName(toolName)
                    .toolStatus("failure")
                    .build();
            // 使用反射设置content字段
            try {
                java.lang.reflect.Field contentField = WorkPanelEvent.class.getDeclaredField("content");
                contentField.setAccessible(true);
                contentField.set(event, errorMessage);
            } catch (Exception e) {
                // 如果反射失败，忽略错误
            }
            
            log.debug("已记录工具调用错误（无对应pending事件）: 工具={}, 错误={}", toolName, errorMessage);
            return event;
        }
    }
    
    /**
     * 生成工具调用键
     * 
     * @param toolName 工具名称
     * @param timestamp 时间戳
     * @return 工具调用键
     */
    private String generateToolCallKey(String toolName, long timestamp) {
        return (toolName != null ? toolName : "unknown") + "_" + timestamp;
    }
    
    /**
     * 查找pending状态的工具调用事件
     * 
     * @param toolName 工具名称
     * @return pending状态的事件，如果未找到则返回null
     */
    private ToolEvent findPendingToolCall(String toolName) {
        if (toolName == null) {
            return null;
        }
        
        // 查找最近的该工具的pending事件
        for (Map.Entry<String, ToolEvent> entry : pendingToolCalls.entrySet()) {
            ToolEvent event = entry.getValue();
            if (toolName.equals(event.getToolName()) && "pending".equals(event.getToolStatus())) {
                return event;
            }
        }
        
        return null;
    }
    
    /**
     * 更新pending事件为完成状态
     * 
     * @param pendingEvent pending状态的事件
     * @param output 工具输出结果
     * @param status 执行状态
     * @param executionTime 执行时间
     * @param currentTime 当前时间
     */
    private void updatePendingEventForCompletion(ToolEvent pendingEvent, Object output, String status, Long executionTime, long currentTime) {
        pendingEvent.setType(WorkPanelUtils.getEventTypeFromStatus(status));
        pendingEvent.setToolOutput(output);
        pendingEvent.setToolStatus(status != null ? status : "unknown");
        pendingEvent.setExecutionTime(executionTime);
        pendingEvent.setTimestamp(currentTime);
    }
    
    /**
     * 创建新的完成事件
     * 
     * @param toolName 工具名称
     * @param output 工具输出结果
     * @param status 执行状态
     * @param executionTime 执行时间
     * @param currentTime 当前时间
     * @return 新的完成事件
     */
    private ToolEvent createNewCompletionEvent(String toolName, Object output, String status, Long executionTime, long currentTime) {
        return ToolEvent.builder()
                .type(WorkPanelUtils.getEventTypeFromStatus(status))
                .timestamp(currentTime)
                .toolName(toolName != null ? toolName : "未知工具")
                .toolOutput(output)
                .toolStatus(status != null ? status : "unknown")
                .executionTime(executionTime)
                .build();
    }
    
    /**
     * 输出工具调用开始日志
     * 
     * @param toolName 工具名称
     * @param input 工具输入参数
     * @param currentTime 当前时间
     */
    private void logToolCallStart(String toolName, Object input, long currentTime) {
        String formattedTime = DATE_FORMAT.format(new java.util.Date(currentTime));
        log.info("\n🔧 工具调用: [{}]\n⏰ 时间: {}\n📥 输入: {}\n📊 状态: 处理中", 
                 toolName != null ? toolName : "未知工具", 
                 formattedTime,
                 WorkPanelUtils.convertToJsonString(input));
    }
    
    /**
     * 输出工具调用完成日志
     * 
     * @param toolName 工具名称
     * @param output 工具输出结果
     * @param status 执行状态
     * @param executionTime 执行时间
     * @param currentTime 当前时间
     */
    private void logToolCallCompletion(String toolName, Object output, String status, Long executionTime, long currentTime) {
        String formattedTime = DATE_FORMAT.format(new java.util.Date(currentTime));
        
        if ("success".equals(status)) {
            log.info("\n🔧 工具调用: [{}]\n⏰ 时间: {}\n✅ 状态: 成功\n📤 输出: {}{}", 
                     toolName != null ? toolName : "未知工具",
                     formattedTime,
                     WorkPanelUtils.convertToJsonString(output),
                     executionTime != null ? "\n⏱️ 耗时: " + executionTime + "ms" : "");
        } else if ("failure".equals(status) || "error".equals(status)) {
            log.info("\n🔧 工具调用: [{}]\n⏰ 时间: {}\n❌ 状态: 失败\n💬 错误: {}{}", 
                     toolName != null ? toolName : "未知工具",
                     formattedTime,
                     WorkPanelUtils.convertToJsonString(output),
                     executionTime != null ? "\n⏱️ 耗时: " + executionTime + "ms" : "");
        }
    }
    
    /**
     * 根据事件类型设置特有属性
     * 
     * @param event 工作面板事件
     * @param data 事件数据
     */
    private void setEventSpecificProperties(WorkPanelEvent event, Map<String, Object> data) {
        String eventType = event.getType();
        
        // 设置通用属性
        if (event instanceof ThoughtEvent) {
            data.put("content", ((ThoughtEvent) event).getContent());
        } else if (event instanceof LogEvent) {
            data.put("content", ((LogEvent) event).getContent());
        } else if (event instanceof ResultEvent) {
            data.put("content", ((ResultEvent) event).getContent());
        }
        
        switch (eventType) {
            case "thought":
                // 思考事件特有属性
                if (event instanceof ThoughtEvent) {
                    ThoughtEvent thoughtEvent = (ThoughtEvent) event;
                    data.put("thinkingType", thoughtEvent.getThinkingType());
                }
                data.put("title", generateTitleForThought(event));
                break;
            case "tool_call":
            case "tool_result":
            case "tool_error":
                // 工具事件特有属性
                if (event instanceof ToolEvent) {
                    ToolEvent toolEvent = (ToolEvent) event;
                    data.put("toolName", toolEvent.getToolName());
                    data.put("toolInput", toolEvent.getToolInput());
                    data.put("toolOutput", toolEvent.getToolOutput());
                    data.put("toolStatus", toolEvent.getToolStatus());
                    data.put("executionTime", toolEvent.getExecutionTime());
                    
                    // 根据状态设置错误信息
                    if ("tool_error".equals(eventType)) {
                        data.put("errorMessage", toolEvent.getErrorMessage());
                        data.put("errorCode", toolEvent.getErrorCode());
                    }
                }
                data.put("title", event instanceof ToolEvent && ((ToolEvent) event).getToolName() != null ? ((ToolEvent) event).getToolName() : "未知工具");
                break;
            case "log":
                // 日志事件特有属性
                if (event instanceof LogEvent) {
                    LogEvent logEvent = (LogEvent) event;
                    data.put("logLevel", logEvent.getLogLevel());
                }
                data.put("title", "日志");
                break;
            case "embed":
                // 嵌入事件特有属性
                if (event instanceof EmbedEvent) {
                    EmbedEvent embedEvent = (EmbedEvent) event;
                    data.put("embedUrl", embedEvent.getEmbedUrl());
                    data.put("embedType", embedEvent.getEmbedType());
                    data.put("embedTitle", embedEvent.getEmbedTitle());
                    data.put("embedHtmlContent", embedEvent.getEmbedHtmlContent());
                }
                data.put("title", event instanceof EmbedEvent && ((EmbedEvent) event).getEmbedTitle() != null ? ((EmbedEvent) event).getEmbedTitle() : "网页预览");
                break;
            case "result":
                // 结果事件特有属性
                data.put("title", "最终答案");
                break;
            default:
                // 默认标题
                data.put("title", generateDefaultTitle(event));
                break;
        }
    }
    
    /**
     * 生成思考事件的标题
     * 
     * @param event 思考事件
     * @return 标题
     */
    private String generateTitleForThought(WorkPanelEvent event) {
        String content = null;
        if (event instanceof ThoughtEvent) {
            content = ((ThoughtEvent) event).getContent();
        } else if (event instanceof LogEvent) {
            content = ((LogEvent) event).getContent();
        } else if (event instanceof ResultEvent) {
            content = ((ResultEvent) event).getContent();
        }
        
        if (content != null && !content.isEmpty()) {
            return content.substring(0, Math.min(content.length(), 50)) + (content.length() > 50 ? "..." : "");
        }
        return "思考过程";
    }
    
    /**
     * 生成默认标题
     * 
     * @param event 事件
     * @return 标题
     */
    private String generateDefaultTitle(WorkPanelEvent event) {
        String content = null;
        if (event instanceof ThoughtEvent) {
            content = ((ThoughtEvent) event).getContent();
        } else if (event instanceof LogEvent) {
            content = ((LogEvent) event).getContent();
        } else if (event instanceof ResultEvent) {
            content = ((ResultEvent) event).getContent();
        }
        
        if (content != null && !content.isEmpty()) {
            return content.substring(0, Math.min(content.length(), 50)) + (content.length() > 50 ? "..." : "");
        }
        return "事件";
    }
    
    /**
     * 处理事件类型特定逻辑
     * 
     * @param event 工作面板事件
     * @param data 事件数据
     */
    private void processEventTypeSpecifics(WorkPanelEvent event, Map<String, Object> data) {
        String eventType = event.getType();
        
        if (eventType != null) {
            switch (eventType) {
                case "tool_call":
                case "tool_result":
                case "tool_error":
                    // 处理工具事件
                    if (event instanceof ToolEvent) {
                        processToolEvent((ToolEvent) event, data);
                    }
                    break;
                case "thought":
                    // 处理思考事件
                    if (event instanceof ThoughtEvent) {
                        processThinkingEvent((ThoughtEvent) event, data);
                    }
                    break;
                // 其他事件类型可以根据需要添加处理逻辑
            }
        }
    }
    
    /**
     * 处理工具事件
     * 
     * @param event 工具事件
     * @param data 事件数据
     */
    private void processToolEvent(ToolEvent event, Map<String, Object> data) {
        if (event == null || data == null) {
            return;
        }
        
        // 添加工具事件特定的日志
        if (log.isInfoEnabled()) {
            log.info("[工具事件] 类型={}, 工具={}, 有toolInput={}, 有toolOutput={}", 
                    event.getType(), 
                    event.getToolName(),
                    event.getToolInput() != null,
                    event.getToolOutput() != null);
        }
        
        if (log.isDebugEnabled()) {
            log.debug("[工具事件详情] toolInput={}, toolOutput={}", 
                    event.getToolInput(),
                    event.getToolOutput());
        }
    }
    
    /**
     * 处理思考事件
     * 
     * @param event 思考事件
     * @param data 事件数据
     */
    private void processThinkingEvent(ThoughtEvent event, Map<String, Object> data) {
        if (event == null || data == null) {
            return;
        }
        
        // 对于最终答案类型的思考事件，添加特殊标记
        if ("final_answer".equals(event.getThinkingType())) {
            data.put("isFinalAnswer", true);
        }
    }
    
    /**
     * 从对象池获取HashMap实例
     */
    private Map<String, Object> acquireMap() {
        return mapPoolService.acquireMap();
    }
    
    /**
     * 获取对象池统计信息
     */
    public String getMapPoolStatistics() {
        return mapPoolService.getMapPoolStatistics();
    }
    
    /**
     * 将HashMap实例归还到对象池
     */
    public void releaseMap(Map<String, Object> map) {
        mapPoolService.releaseMap(map);
    }
}